/*
 * ZVMTasks.java
 * Created on June 14, 2005, 12:41 PM
 */

package live.zvmtasks;

import java.util.LinkedList;
import java.util.StringTokenizer;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Collections;
import java.util.Collection;
import java.util.Vector;
import java.math.BigDecimal;
import live.smapi.SMAPI_Interface;
import live.smapi.exceptions.*;
import live.dto.Interval;
import live.dto.SavedSegment;
import live.dto.Guest;
import live.dto.Network;
import live.dto.Group;
import live.dto.MDisk;
import live.dto.NameList;
/**
 * ZVMTasks is a library of functions that perform some user oriented task on
 * a z/VM system.  It makes extensive use of the SMAPIWrapper library.
 *
 * @author Jason J. Herne 
 */
public class ZVMTasks
{
    private SMAPI_Interface smapi;
    /*private String userName;
    private String password;
    private String hostName;
    private int port;*/
    
    public static final String	PROFILE_GENERAL =   "LIVEP0";
    public static final String	PROFILE_SUPER	=   "LIVEP1";
    
    public static final String SEG_BUILDER	=   "LIVESEGB";
    
    /** 
     * Creates a new instance of ZVMTasks.
     * @param hostName Name or number of the host
     * @param port Port number
     * @param userName Authenticated user name
     * @param password User's password
     */
    public ZVMTasks(String hostName, int port, String userName, String password)
    {
	/*setHostName(hostName);
	setPort(port);
	setUserName(userName);
	setPassword(password);*/
	
	smapi = new SMAPI_Interface(hostName, port, userName, password);
    }
    
    /**
     * Sets the host name or IP address of the z/VM system we will connect to.
     *
     * @param	hostName The host name or IP address  of the z/VM system to connect to.
     */
    /*public void setHostName(String hostName)
    {
	this.hostName = hostName;
    }*/
    
    /**
     * Sets the port number the SMAPI server is listening on.
     *
     * @param	port The port number the SMAPI server is listening on.
     */
    /*public void setPort(int port)
    {
	this.port = port;
    }*/
    
    /**
     * Sets the user name used to authenticate to the SMAPI server.
     *
     * @param	userName The user name used to authenticate to the SMAPI server.
     */
    /*public void setUserName(String userName)
    {
	this.userName = userName;
    }*/
    
    /**
     * Sets the password used to authenticate to the SMAPI server.
     *
     * @param	password The password used to authenticate to the SMAPI server.
     */
    /*public void setPassword(String password)
    {
	this.password = password;
    }*/

    /**
     * Tests whether this user can authenticate to the SMAPI server.
     * Wraps (@link live.smapi.SMAPI_Interface#nullRequest()}
     *
     * @return	True if authentication succeeded.
     */
    public boolean authenticate()
	throws ZVMTasksException 
    {
	try
	{
	    smapi.nullRequest();
	}
	catch(RequestNotAuthorizedException e)
	{
	    /* We did not authenticate. */
	    return false;
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	// If we get here, the null request completed ok, authentication good :)
	return true;
    }
    
    /**
     * Returns a linkedlist of Networks.  Each Network is populated with its name 
     * and network type (Guest LAN or VSwitch)
     * Wraps {@link live.smapi.SMAPI_Interface#virtualLanQuery()}
     *
     * @return	{@link java.util.Linkedlist} of {@link live.dto.Network}s on the z/VM system.
     */
    public LinkedList queryAllNetworks()
	throws ZVMTasksException
    {
	LinkedList networks = null;
	
	try
	{
	    networks = smapi.virtualLanQuery();
	}
	catch(LANDoesNotExistException e)
	{
	    //If there are no LANs, just return an empty list.
	    networks = new LinkedList();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	return networks;
    }
        
    /**
     * Queries all the guests on the system.
     * Returns a {@link java.util.LinkedList} of {@link java.lang.String}s.  Each String is the name of a guest on
     * the system.
     * Wraps {@link live.smapi.SMAPI_Interface#imageNameQuery()}
     *
     * @return	{@link java.util.LinkedList} of guest names on the z/VM system.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public LinkedList queryAllGuestNames()
	throws ZVMTasksException
    {
	LinkedList guests = null;
	
	try
	{
	    guests = smapi.imageNameQuery();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	return guests;
    }
    
    /**
     *  Creates a new Guest LAN and attaches a guest to it with a virtual NIC of the specified address
     *  Wraps {@link live.smapi.SMAPI_Interface#virtualLANCreateAndConnectDM(String, String, String)}
     *   & {@link live.smapi.SMAPI_Interface#virtualLANCreateAndConnect(String, String)}.  
     *
     * @param lanName Name of LAN to create
     * @param guestName Name of guest to connect
     * @param vaddr Virtual address to create NIC at
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void guestLANCreateConnect(String lanName, String guestName, String vaddr)
	throws ZVMTasksException
    {
	try
	{
	    smapi.virtualLANCreateAndConnectDM(lanName, guestName, vaddr);
	    smapi.virtualLANCreateAndConnect(lanName, guestName);
	}
	catch(ImageNotActiveException e)
	{
	    //DO nothing, the guest is logged off, next time he logs on, the adapter
	    //will be created.
	}
	
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Creates a new Guest LAN with no one attached to it
     * {@link live.smapi.SMAPI_Interface#virtualLANCreate(String)}
     *
     * @param lanName Name of LAN to create
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void guestLANCreate(String lanName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.virtualLANCreate(lanName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Disconnects a guest (with a specified NIC address) from a LAN and deletes the LAN if it is the last guest on the LAN
     *  Wraps {@link live.smapi.SMAPI_Interface#virtualLANDisconnectAndDeleteDM(String, String, String)}
     *   & {@link live.smapi.SMAPI_Interface#virtualLANDisconnectAndDelete(String, String, String)}
     *
     * @param lanName Name of LAN to disconnect
     * @param guestName Name of guest to disconnect
     * @param vaddr Virtual address of guest's NIC
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void guestLANDeleteDisconnect(String lanName,  String guestName, String vaddr)
	throws ZVMTasksException
    {
	try
	{
	    smapi.virtualLANDisconnectAndDeleteDM(lanName, guestName, vaddr);
	    smapi.virtualLANDisconnectAndDelete(lanName, guestName, vaddr);
	    
	}
	catch(ImageNotActiveException e)
	{
	    //DO nothing, the guest is logged off, next time he logs on, the adapter
	    //will not exist.
	}
	
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Creates a new VSwitch with no one attached to it
     *  Wraps {@link live.smapi.SMAPI_Interface#virtualSwitchCreate(String, String, String)}
     *
     * @param lanName Name of VSwitch to create
     * @param portname Name of real port for VSwitch (specified if you want to connect the VSwitch externally)
     * @param raddr Real address of port (specified if you want to connect the VSwitch externally)
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void virtualSwitchCreate(String lanName, String portname, String raddr)
	throws ZVMTasksException
    {
	try
	{
	    smapi.virtualSwitchCreate(lanName, portname, raddr);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Creates a new VSwitch and attaches a guest to it with a virtual NIC of the specified address
     * Wraps {@link live.smapi.SMAPI_Interface#virtualSwitchCreateAndConnectDM(String, String, String, String, String)}
     *  & {@link live.smapi.SMAPI_Interface#virtualSwitchCreateAndConnect(String, String, String, String, String)}. 
     *
     * @param lanName Name of VSwitch to create
     * @param guestName Name of guest to connect
     * @param vaddr Virtual address to create NIC at
     * @param portname Name of real port for VSwitch
     * @param raddr Real address of port (specified if you want to connect the VSwitch externally)
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
        public void virtualSwitchCreateConnect(String lanName, String guestName, String vaddr, String portname, String raddr)
	throws ZVMTasksException
    {
	try
	{
	    //DM version should be done first, else we might hop out in case of image not active exception.
	    smapi.virtualSwitchCreateAndConnectDM(lanName, guestName, vaddr,  portname, raddr);
	    smapi.virtualSwitchCreateAndConnect(lanName, guestName, vaddr,  portname, raddr);
	}
	catch(ImageNotActiveException e)
	{
	    //DO nothing, the guest is logged off, next time he logs on, the adapter
	    //will be created.
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	try
	{
	    //Must ensure the guest is added to the Access List for this VSWITCH.
	    smapi.virtualSwitchSet(lanName, guestName, ""); 
	}
	catch(AccessAlreadyGrantedException e)
	{
	    //Do nothing.  The guest is already on the access list for this network.
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Detaches a guest from a VSwitch
     * Wraps{@link live.smapi.SMAPI_Interface#virtualSwitchDisconnectAndDeleteDM(String, String, String)}
     * & {@link live.smapi.SMAPI_Interface#virtualSwitchDetatch(String, String)}
     *
     * @param switchName Name of VSwitch
     * @param guestName Name of guest to detach
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void virtualSwitchDetatch(String switchName, String guestName)
	throws ZVMTasksException
    {
	String vaddr = "";
	
	try
	{
	    //Get the virtual addres of the NIC connected to this network.
	    vaddr = smapi.getNICaddress(switchName, guestName);
	    
	    //Delete the guests NICDEF directory statement for this network.
	    smapi.virtualSwitchDisconnectAndDeleteDM(guestName, switchName, vaddr);
	}
	//catch(SMAPIException e)
	//{
	// Do we want to catch 'def does not exist exception' and do nothing?    
	//}
	catch(SMAPIException e)
	{
	    //If we get here, getting the vaddr of the NIC failed.
	    throw new ZVMTasksException(e);
	}
	
	try
	{
	    smapi.virtualSwitchDetatch(guestName, vaddr);
	}
	catch(SMAPIException e)
	{
	    //throw new ZVMTasksException(e);
	}
	
	//Remove guest from vswitch's auth list.
	//We do this FIRST because the switch will not exist if there is only 1 guest
	//left connected and we try to do it AFTER we remove him.
	virtualSwitchRevoke(switchName, guestName);
	
    }
    
    /**
     *  Deletes a VSwitch, even if there are people connected to it
     * Wraps {@link live.smapi.SMAPI_Interface#virtualSwitchDelete(String)}
     *
     * @param switchName Name of VSwitch to delete
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void virtualSwitchDelete(String switchName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.virtualSwitchDelete(switchName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Restricts a user from being on a VSwitch's authorized list
     * Wraps {@link live.smapi.SMAPI_Interface#virtualSwitchSet(String, String, String)}
     *
     * @param switchName Name of VSwitch 
     * @param userName Name of guest to restrict
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void virtualSwitchRevoke(String switchName, String userName)
       throws ZVMTasksException
    {
	try
	{
	    smapi.virtualSwitchSet(switchName, "&", userName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Grants a user (already connected) access to a VSwitch
     * Wraps {@link live.smapi.SMAPI_Interface#virtualSwitchSet(String, String, String)}
     *
     * @param switchName Name of VSwitch
     * @param userName Name of guest to grant access
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void virtualSwitchGrant(String switchName, String userName)
       throws ZVMTasksException
    {
	try
	{
	    smapi.virtualSwitchSet(switchName, userName, "&");
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     *  Finds the lowest block of address space available for use as a virtual NIC
     *  Wraps {@link live.smapi.SMAPI_Interface#getLowestFreeAddress(String, String)}
     * NOT CURRENTLY USED
     *
     * @param strGuestName name of guest to query
     * @return 4 digit hexadecimal address of start of free address block
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public String findNICaddress(String strGuestName)
	throws ZVMTasksException
    {
	String i;
	try
	{
	    i=smapi.getLowestFreeAddress(strGuestName, "3");
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	return i;
    }
    
    /**
     *  Queries a network for a specified guest and returns that guest's virtual NIC
     * Wraps {@link live.smapi.SMAPI_Interface#getNICaddress(String, String)}
     *
     * @param strNetworkName Network to query
     * @param strGuestName Name of guest to query for
     * @return 4 digit hexadecimal address of that guest's virtual NIC on that network
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public String getNICaddress(String strNetworkName, String strGuestName)
	throws ZVMTasksException
    {
	String i;
	try
	{
	    i=smapi.getNICaddress(strNetworkName, strGuestName);
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	return i;
    }
    
    
    /*
     * Allows creation of multiple networks or connections to networks.  A list
     * of Network objects is passed as argument and a list of TaskItem objects
     * is returned.
     */
    /*public LinkedList createConnectMultipleNetworks(LinkedList networks)
	throws ZVMTasksException
    {
	LinkedList taskItems = new LinkedList();
	
	ListIterator itr= networks.listIterator();
	Network curNet = null;
	while(itr.hasNext())
	{
	    Object result = null;
	    
	    try
	    {
		curNet = ((Network)itr.next());
		if(curNet.getNetworkType() == Network.NETWORK_GLAN)
		{
		    guestLANCreateConnect( curNet.getNetworkName(), curNet.connectedGuest, curNet.connectedVaddr);
		}
		else //Network is a VSwitch.
		{
		    virtualSwitchCreateConnect(curNet.getNetworkName(), curNet.connectedGuest, curNet.connectedVaddr, curNet.portname1, curNet.rdev1);
		}
	    }
	    catch( Exception e)
	    {
		result = e;
	    }
	    
	    //Now that the operator is complete, lets figure out if there was an error.
	    //If so, we'll record it, if not, record an "All ok" message :)
	    String taskName = curNet.getNetworkName() + " <-- " + curNet.connectedGuest;
	    if(result == null) result = "Task Completed without error.";
	    
	    taskItems.add( new TaskItem(taskName, result));
	}
	
	return taskItems;
    }*/
    
    /**
     *  Returns a {@link java.util.LinkedList} of all networks and their members
     *  Wraps {@link live.smapi.SMAPI_Interface#queryNetworkMembers(String)}
     *
     * @return {@link java.util.LinkedList} of {@link live.dto.Network}s each containing network name and a list of members
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public LinkedList getAllNetworkNames()
     throws ZVMTasksException
    {
	LinkedList networks = null;
	
	try
	{
	    networks = smapi.queryNetworkMembers("");
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}

	return networks;
    }
    
    /**
     *  Queries the members of a particular VSwitch
     *  Wraps {@link live.smapi.SMAPI_Interface#queryNetworkMembers(String)}.
     *
     * @param strNetName Name of VSwitch to query
     * @return A {@link live.dto.Network} containing network name and list of members
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public Network getNetworkMembers(String strNetName)
     throws ZVMTasksException
    {
	Network curnet = null;
	LinkedList networks = null;
	
	try
	{
	    networks = smapi.queryNetworkMembers(strNetName);
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	curnet = (Network)networks.removeFirst();
	return curnet;
    }

    /**
     *  Queries the information for a given network
     *   Wraps {@link live.smapi.SMAPI_Interface#virtualLanQuery()}.
     *
     * @param NetName Name of network to query
     * @return A {@link live.dto.Network} with information on the queried network
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public Network getNetworkInfo(String NetName)
     throws ZVMTasksException
    {
	LinkedList netinfo = null;
	Network thisnetwork = null;
	try
	{
	    netinfo = smapi.virtualLanQuery();
	    if (netinfo.size()==1)
		thisnetwork = (Network) netinfo.getFirst();
	    else
		throw new ZVMTasksException("Should not have returned more than one network with the same name!");
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	return thisnetwork;
    }
    
    /**
     * @see live.threads.ThreadedZVMTasks.FindCommonFreeAddress
     */
    public static interface getCommonAddressProgressListener
	{
	    public void anotherComplete();
	}
    
    /**
     * Overloaded call of the original
     *
     * @param memberNames List of guests to find the common address among
     * @return lowest 4-digit hexadecimal virtual address available
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public String getCommonAddress(Collection memberNames)
	throws ZVMTasksException
    {
    	return getCommonAddress(memberNames, null);
    }
    
    /**
     *  Takes a list of guests and finds the lowest available common NIC address amongst them
     *
     * @param memberNames LinkedList of guests to find lowest address among
     * @return 4 digit hexadecimal string of lowest common free address
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     * @throws NoCommonAddressException Could not find a common address among the given guests
     */
    public String getCommonAddress(Collection memberNames, getCommonAddressProgressListener listener)
	throws ZVMTasksException
    {
	LinkedList AllAddrs = getAllFreeAddresses(memberNames, listener);
		    
	ListIterator iterAll = AllAddrs.listIterator(); //Go through big list of free addresses
	LinkedList oneUsersList = (LinkedList)iterAll.next(); //Go through each user's list of free addresses
	ListIterator iterone = oneUsersList.listIterator();
	boolean found_comm = false;
	String commonaddr = (String)iterone.next();
	    
	//Find a common low address
	while(iterone.hasNext() && !found_comm)
	{
	    found_comm=true;
	    iterAll = AllAddrs.listIterator(); //Go through big list of free addresses
	    //Cycle through all the lists and see if everybody has the same
	    while(iterAll.hasNext() && found_comm)
	    {
		LinkedList anotherUsersList = (LinkedList)iterAll.next();
		//if that user doesn't have that address free, set found_comm to false, so we'll 
		if (!anotherUsersList.contains(commonaddr)) found_comm=false;
	    }
	    if (!found_comm) commonaddr=(String)iterone.next();
	 }
	//If we'return at the end of our list and we have not found a common address
	// Throw an exception that lists all the members we were trying to find an address for
	if(!iterone.hasNext() && !found_comm) 
	{
	    commonaddr = null;
	    throw new NoCommonAddressException();
	}
	return commonaddr;
    }
    
    /**
     * Takes a list of member names and returns a {@link java.util.LinkedList} of {@link java.util.LinkedList}s of all their
     * free addresses (gaps being bigger than 3 addresses so we can use the addresses for 
     * virtual NIC addresses)
     *
     * @param memberNames LinkedList of strings (guest names)
     * @return {@link java.util.LinkedList} of {@link java.util.LinkedList}s of free addresses (provided there are 3+ contiguous free addresses in a row
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public LinkedList getAllFreeAddresses(Collection memberNames, getCommonAddressProgressListener listener)
	throws ZVMTasksException
    {
	LinkedList AllAddrs = new LinkedList();
	Iterator iterinp = memberNames.iterator();
	    
	//Go through all names, and build a List of all their Lists of free addresses
	while(iterinp.hasNext())
	{
	    LinkedList AllFreeAddrs = new LinkedList(); //list of free addresses
	    String userName = (String)iterinp.next();
	    LinkedList addrlist = getStorageGaps(userName);//get each user's free intervals
	    ListIterator onelistiter = addrlist.listIterator();
			
	    //Go through one list and flesh out the addresses so we have a list of all available addresses
	    while(onelistiter.hasNext())
	    {
		    Interval interv = (Interval)onelistiter.next();
		    Integer addr = Integer.decode("0x"+interv.lower);
		    Integer roof = Integer.decode("0x"+interv.upper);
			    
		    int adr = addr.intValue();
		    int rf = roof.intValue();
			    
		    //Add all free addresses to list (expand from range to list of all available) 
		    while (adr < (rf-2)) //stop 2 below upper so that we have a block of 3 free addresses
		    {
			AllFreeAddrs.add(Integer.toHexString(adr));
			adr++;
		    }
			
	    }
	    //Add each list of free addresses to the big list of free addresses
	    AllAddrs.add(AllFreeAddrs);
	    
	    if(listener != null)
	    	listener.anotherComplete();
	}
	return AllAddrs;
    }
    
    /**
     *  Returns all the virtual address gaps for a user
     *   Wraps {@link live.smapi.SMAPI_Interface#returnQueryVirtual(String)}.
     *
     * @param imageName Name of guest to query
     * @return {@link java.util.LinkedList} of {@link live.dto.Interval}s with the address space gaps
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public LinkedList getStorageGaps(String imageName)
	throws ZVMTasksException
    {
	LinkedList query; // = new LinkedList();
	
	//Get Query Virtual
	try
	{
	    query = smapi.returnQueryVirtual(imageName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException (e);
	} 
	
	//Parse the linkedlist returned from QueryVirtual, get all addresses -- addrtype = 1
	Vector addresses = parseQueryVirtual(query, 1);

	//Iterate through the list and find the sizes of all the gaps in used addresses
	ListIterator iter2 = addresses.listIterator();
	LinkedList gaps = new LinkedList();
	String addr1 = (String)iter2.next();
	Integer a1 = Integer.decode(addr1);
	if (!iter2.hasNext())
	{
	    Interval inter = new Interval(Integer.toHexString(a1.intValue()+1), Integer.toHexString(0xFFFF));
	    gaps.addLast(inter);
	}
	while (iter2.hasNext())
	{

	    String addr2=(String)iter2.next();
	    //Convert Strings to Integers to ints
	    // yes it's very long, but the hexadecimal strings are hard to work with
	    a1 = Integer.decode(addr1);
	    Integer a2 = Integer.decode(addr2);
	    int iad = a2.intValue()- a1.intValue();
	    int a11 = a1.intValue()+ 1;
	    
	    //If the gap is big enough to use, add it to the list
	    if (iad > 3) 
	    {
		Interval inter = new Interval(Integer.toHexString(a11), Integer.toHexString(a2.intValue()));
		gaps.addLast(inter);
	    }
	    addr1=addr2;
	    if (!iter2.hasNext())
	    {
		Interval inter = new Interval(Integer.toHexString(a2.intValue()+1), Integer.toHexString(0xFFFF));
		gaps.addLast(inter);
	    }
	}

	return gaps;
    }
    
    /**
     *  Sorts through the results of query virtual, and returns a vector of sorted addresses 
     *   with no duplicates
     *
     * @param queryRes Results of query virtual
     * @param addrtype 1 for all addresses, 2 for NIC addresses, 3 for DASD
     * @return {@link java.util.Vector} of addresses in 4 digit hexadecimal notation (i.e. 0x0009)
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    private Vector parseQueryVirtual(LinkedList queryRes, int addrtype)
	throws ZVMTasksException
    {
	Vector addresses = new Vector();
	ListIterator iter = queryRes.listIterator();
	boolean isRightType = false;
	final String NICS = "NICS";
	final String DASD = "DASD";
	final String OTHER = "OTHER";
	
	//Go through the list of addresses
	while (iter.hasNext())
	{
	    String token = (String)iter.next();
	    //Take only the addresses, throw away the other stuff
	    StringTokenizer tokaddr = new StringTokenizer(token);
	    String addr = tokaddr.nextToken();
	    //Decide which addresses to take based on what the caller wants
	    switch(addrtype)
	    {
		//Take all addresses
		case 1:
		    if (addr.equals(OTHER) || addr.equals(DASD) || addr.equals(NICS))
		    {}
		    else
			addresses.add("0x" + addr);
		    break;
		
		//Get only the virtual NICs
		case 2:
		    if (addr.equals(NICS))
			isRightType = true;
		    else if(isRightType)
			addresses.add("0x" + addr);
		    break;
		
		//Get only the DASD
		case 3:
		    if (addr.equals(DASD))
			isRightType = true;
		    else if (addr.equals(NICS))
			isRightType = false;
		    else if(isRightType)
			addresses.add("0x" + addr);
		    break;
		
		default:
		    break;
	    }
	}
	
	//Go through and find all duplicates
	Iterator iter2 = addresses.iterator();
        String addr1 = (String)iter2.next();
	String addr2 = null;
	if (iter2.hasNext()) addr2 = (String)iter2.next();
	
	while (iter2.hasNext())
	{
	    //Go through and delete all duplicates
	    if (addr1.equals(addr2))
	    {
	        iter2.remove();
	    }
	    addr1=addr2;
	    addr2=(String)iter2.next();
	}
	
	//Sort everything and return
	Collections.sort(addresses);
	return addresses;
    }

   /**
     *  Removes/adds a guest to/from a group.  
     *  This will create a new group if necessary and will delete a group if the last member is 
     *   removed.  All group names have the format LIVE_grpname  
     *   Wraps {@link live.smapi.SMAPI_Interface#nameListRemove(String, String)}
     *     &  Wraps {@link live.smapi.SMAPI_Interface#nameListAdd(String, String)}.
     *
     * @param imageName Name of guest to add/remove
     * @param oldGroupName Name of group to remove guest from (optional)
     * @param newGroupName Name of group to add guest to (optional)
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void setGuestsGroup(String imageName, String oldGroupName, String newGroupName)
	throws ZVMTasksException
    {
    	try
	{
	    if(oldGroupName != null)
		smapi.nameListRemove("LIVE_" + oldGroupName, imageName);
	    if(newGroupName != null)
		smapi.nameListAdd("LIVE_" + newGroupName, imageName);
	}
    	catch(Exception e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
   /**
     *  Deletes a guest
     *    Wraps {@link live.smapi.SMAPI_Interface#imageDelete(String)}.
     *
     * @param imageName Name of guest to delete
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void deleteImage(String imageName)
	throws ZVMTasksException
    {
    	try
	{
	    smapi.imageDelete(imageName);
	}
    	catch(Exception e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     * This function gets all the namelist names and contents from the server and puts them
     * in a list of {@link live.dto.Group} objects
     *   Wraps {@link live.smapi.SMAPI_Interface#nameListQueryAll()}.
     *
     * @return A {@link java.util.LinkedList} of {@link live.dto.Group}s.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
	public LinkedList getGroups()
	    throws ZVMTasksException
	{
		LinkedList nameLists = null;
		LinkedList groups = new LinkedList();
		
		try
		{
			nameLists = smapi.nameListQueryAll();
			
			Iterator iter = nameLists.iterator();
			while(iter.hasNext())
			{
				// For each list in nameLists...
				NameList list = (NameList) iter.next();
				
				try
				{
					// check to see if its name starts with LIVE_
					// TODO: break "LIVE_" and the magic number 5 off somewhere into a constant
					if(list.getListName().substring(0,5).equals("LIVE_"))
					{
						// And if so, it's one of our groups.
						Group group = new Group(list.getListName().substring(5));
						
						// So go through each of the members...
						Iterator guestIter = list.iterator();
						while(guestIter.hasNext())
							// and add them to the current group
							group.addMember( (String) guestIter.next() );
						
						// When that's done, the current group to our collection of groups
						groups.add(group);
					} // if the list is one of our groups
				}
				// This exception is no biggie, in fact is probably not an error at all.
				// It just means that we're looking at a name list that is shorter than 5
				// characters
				catch(IndexOutOfBoundsException e)
				{ }
					
			} // while iter.hasNext; foreach list in nameLists
		} // try
		catch(SMAPIException e)
		{
			throw new ZVMTasksException(e);
		}
		
		return groups;
    }
    
    /**
     * This function queries the characteristics of a guest 
     *  Wraps {@link live.smapi.SMAPI_Interface#imageQuery(String)}.
     *
     * @param guestName Name of guest to query
     * @return A {@link java.util.Guest}
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public Guest queryGuest(String guestName)
	throws ZVMTasksException
    {
	LinkedList lines = null;
		
	try
	{
	    lines = smapi.imageQuery(guestName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	//We have no way of getting the group name so we just set it as null.
	//It is the responsibility of the caller of this function to know that
	//he must not try and use the group parameter of the guest object returned.
	Guest guest = new Guest(guestName, null);
	
	StringTokenizer strTok = new StringTokenizer((String)lines.get(0));
	if (strTok.countTokens() < 3) throw new ZVMTasksException("Not enough tokens in USER line!");

	strTok.nextToken(); //Eat 'USER'
	strTok.nextToken(); //Eat guests name
	strTok.nextToken(); //Eat guests pasword
	
	String initMemory;
	String maxMemory;
	String permissionClasses;
	
	//Some guests have no more tokens on this line!
	if(strTok.countTokens() >= 3)
	{
	    initMemory = strTok.nextToken();
	    maxMemory = strTok.nextToken();
	    permissionClasses = strTok.nextToken();
	}
	else
	{
	    initMemory = "";
	    maxMemory = "";
	    permissionClasses = "";
	}
	
	
	guest.setPermissionClasses(permissionClasses);
	guest.setInitMem(initMemory);
	guest.setMaxMem(maxMemory);
	
	return guest;
    }
    
   /**
     * This function returns a {@link java.util.LinkedList} of all active guests
     *  Wraps  {@link live.smapi.SMAPI_Interface#activeImagesQuery()}.
     *
     * @return {@link java.util.LinkedList} of guest names.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public LinkedList activeImagesQuery()
	throws ZVMTasksException
    {
	LinkedList images = null;
	
	try
	{
	    images = smapi.activeImagesQuery();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	return images;
    }
    
    /**
     * This function activates a guest 
     * Wraps  {@link live.smapi.SMAPI_Interface#imageActivate(String)}.
     *
     * @param imageName Name of guest to activate
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void imageActivate(String imageName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.imageActivate(imageName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
   /**
     * This function deactivates a guest 
     * Wraps  {@link live.smapi.SMAPI_Interface#imageDeactivate(String)}.
     *
     * @param imageName Name of guest to deactivate
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void imageDeactivate(String imageName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.imageDeactivate(imageName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
   /**
     * This function sets a guest's password  
     * Wraps  {@link live.smapi.SMAPI_Interface#imageSetPassword(String, String)}.
     *
     * @param strGuestName Name of guest to change password of
     * @param strPassword New Password 
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */
    public void setGuestPassword (String strGuestName, String strPassword)
	throws ZVMTasksException
    {
	try
	{
	    smapi.imageSetPassword(strGuestName, strPassword);
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     * This function creates a new minidisk for a particular user
     * Wraps  {@link live.smapi.SMAPI_Interface#imageDiskCreate(String, String, String, String, String, String, String, String, String, String)}.
     * Wraps  {@link live.smapi.SMAPI_Interface#imageDiskCreateDM(String, String, String, String, String, String, String, String, String, String)}.
     *
     * @param strGuestName Name of guest to create minidisk for
     * @param strVirtualDeviceAddress Virtual device address of minidisk
     * @param strAllocationType How to allocate the minidisk (group region volume or temporary)
     * @param strDiskSize Size of minidisk to create
     * @param strDiskMode Access mode for minidisk (read, write, etc)
     * @param strReadPW Read access password
     * @param strWritePW Write access password
     * @param strMultiPW Multi access password
     * @throws ZVMTasksException An error occurred, just passes up SMAPIException
     *
     */
    public void createMDisk (String strGuestName, String strVirtualDeviceAddress, 
    		String strAllocationType, String strDiskSize, String strDiskMode, 
		String strReadPW, String strWritePW, String strMultiPW)
	throws ZVMTasksException
    {
	String strAllocUnitSize = null;
	if (strAllocationType.equals("T-DISK") | strAllocationType.equals("V-DISK"))
	    strAllocUnitSize = "BLK4096";
	else
	    strAllocUnitSize = "CYLINDERS";
	MDisk thisDisk = new MDisk(strVirtualDeviceAddress, strAllocationType, "0", strDiskSize, "LVDASD", strDiskMode, true, strReadPW, strWritePW, strMultiPW, strAllocUnitSize);
	strDiskSize = convertSizeToCylinders(thisDisk);
	try
	{
	    smapi.imageDiskCreateDM(strGuestName, strVirtualDeviceAddress, strAllocationType, strAllocUnitSize, strDiskSize, strDiskMode, "CMS", strReadPW, strWritePW, strMultiPW);
	    smapi.imageDiskCreate(strGuestName, strVirtualDeviceAddress, strAllocationType, strAllocUnitSize, strDiskSize, strDiskMode, "CMS", strReadPW, strWritePW, strMultiPW);
	}
	catch (ImageDiskInUseException e) {}
	catch (ImageNotActiveException e)
	{
	    //Do nothing, we're already creating it as a directory object too
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     * Converts the size of a disk from megabytes to numbers of cylinders or blocks
     *
     * @param disk information for the Minidisk that you are working with
     * @return Size in blocks or cylinders
     */
    public String convertSizeToCylinders(MDisk disk)
    {
	int cursize = 0;
	//TDisks are in block sizes of 4096 bytes
	if (disk.getAllocationType().equals("T-DISK"))
	{
	    cursize = Integer.parseInt(disk.getSize()) * 1024 *1024;
	    cursize = cursize/4096 + 1;
	}
	//VDisks are in block sizes of 512 bytes
	else if(disk.getAllocationType().equals("V-DISK"))
	{
	    cursize = Integer.parseInt(disk.getSize()) * 1024 *1024;
	    cursize = cursize/512 + 1;
	}
	//3390 DASD is in cylinders of size 850KB
	else
	{
	    cursize = Integer.parseInt(disk.getSize()) * 1024;
	    cursize = cursize/(850) + 1;
	}

	String newsize = Integer.toString(cursize);
	return newsize;
    }
    
   /**
     * This function deletes a minidisk for a particular user
     * Wraps {@link live.smapi.SMAPI_Interface#imageDiskDeleteDM(String, String)}
     * Wraps {@link live.smapi.SMAPI_Interface#imageDiskDelete(String, String)}.
     *
     * @param strGuestName Name of guest to create minidisk for
     * @param strVirtualDeviceAddress Virtual device address of minidisk
     * @throws ZVMTasksException An error occurred, just passes up SMAPIException
     */
    public void deleteMDisk (String strGuestName, String strVirtualDeviceAddress)
	throws ZVMTasksException
    {
	try
	{
	    smapi.imageDiskDeleteDM(strGuestName, strVirtualDeviceAddress);
	    smapi.imageDiskDelete(strGuestName, strVirtualDeviceAddress);
	}
	catch (ImageNotActiveException e)
	{
	    //Do nothing, we're already deleting its directory entry
	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
   /**
     * This function queries minidisks for a particular user
     * Wraps  {@link live.smapi.SMAPI_Interface#imageQuery(String)}.
     *
     * @param strGuestName Name of guest to create minidisk for
     * @return {@link java.util.LinkedList} of {@link live.dto.MDisk}s
     * @throws ZVMTasksException An error occurred, just passes up SMAPIException
     */
    public LinkedList queryMDisks (String strGuestName)
	throws ZVMTasksException
    {
	LinkedList guestDir; // = new LinkedList();
	try
	{
	    guestDir= smapi.imageQuery(strGuestName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	Collections.sort(guestDir);
	//Iterate through existing GuestDirectory
	ListIterator iter = guestDir.listIterator();
	LinkedList mDisks =  new LinkedList();
	
	//Go through list
	while(iter.hasNext())
	{
	    //Tokenize each line in the user directory
	    StringTokenizer stringTok = new StringTokenizer((String)iter.next());
	    String firstWord = stringTok.nextToken();
	    
	    //if it's a minidisk statement, parse it out
	    if(firstWord.equals("MDISK"))
	    {
		MDisk thisDisk = new MDisk();
		//remove it from the directory
		String addr = stringTok.nextToken();

		//First get the virtual address
		thisDisk.setVirtAddr(addr);	
		    
		//Get the device type (3390, TDISK, VDISK)
		String devType = stringTok.nextToken();
		//3390 is the device type for DASD, which is specified in cylinders
		if (devType.equals("3390"))
		    thisDisk.setBlOrCyl("CYLINDERS");
		    
		//get the starting number for the minidisk
		thisDisk.setStartLoc(stringTok.nextToken());

		//get the size of the minidisk
		String amt = stringTok.nextToken();
		    
		//the size could be specified as the entire rest of the available space
		if (amt.equals("END"))
		{
		    //Cylinders are one size
		    if (thisDisk.getBlOrCyl().equals("CYLINDERS"))
		    {
			int amount = 1113 - Integer.parseInt(thisDisk.getStartLoc());
			thisDisk.setSize(Integer.toString(amount));
		    }
		    //VDisks are another size
		    else
		    {
			int amount = 4194296 - Integer.parseInt(thisDisk.getStartLoc());
			thisDisk.setSize(Integer.toString(amount));
		    }
		}
		//or the size is actually just a regular size
		else
		    thisDisk.setSize(amt);
		    
		thisDisk.setSize(convertSizetoMB(thisDisk));
		//if it's not a TDisk or VDisk then the next token will be the volume ID that the DASD's coming from
		if (stringTok.hasMoreTokens() && !(thisDisk.getStartLoc().equals("V-DISK") | thisDisk.getStartLoc().equals("T-DISK")))
		{
		    thisDisk.setVolumeID(stringTok.nextToken());
		}
		//if it IS a TDisk or VDisk the next token is access mode
		else
		{
		    thisDisk.setMode(stringTok.nextToken());
		    if (stringTok.hasMoreTokens())
			thisDisk.setHasPW(true);
		}
		    
		//For DASD only, access mode
		if (stringTok.hasMoreTokens())
		{
		    thisDisk.setMode(stringTok.nextToken());
		    if(stringTok.hasMoreTokens())
		    {
			thisDisk.setHasPW(true);
			thisDisk.setReadPW(stringTok.nextToken());
			if (stringTok.hasMoreTokens()) thisDisk.setWritePW(stringTok.nextToken());
			if (stringTok.hasMoreTokens()) thisDisk.setMultiPW(stringTok.nextToken());
		    }
		}
		//Add it to our list of disks
		mDisks.add(thisDisk);
	    }
	}
	return mDisks;
    }
    
    /**
     * Converts the size of a minidisk from blocks or cylinders to MB
     *
     * @param disk {@link live.dto.MDisk} to convert size of
     * @return String size of minidisk in MB
     */
    public String convertSizetoMB(MDisk disk)
    {
	BigDecimal cursize = new BigDecimal(disk.getSize());
	BigDecimal fourK = new BigDecimal(4096);
	BigDecimal oneK = new BigDecimal(1024);
	BigDecimal halfK = new BigDecimal(512);
	BigDecimal oneCyl = new BigDecimal(850);
	BigDecimal thousandK = oneK.multiply(oneK);
	if (disk.getAllocationType().equals("T-DISK"))
	{
	    cursize = cursize.multiply(fourK);
	    cursize = cursize.divide(thousandK, 1, BigDecimal.ROUND_UP);
	}
	else if(disk.getAllocationType().equals("V-DISK"))
	{
	    cursize = cursize.multiply(halfK);
	    cursize = cursize.divide(thousandK, 1, BigDecimal.ROUND_UP);
	}
	else 
	{
	    cursize = cursize.multiply(oneCyl);
	    cursize = cursize.divide(oneK, 1, BigDecimal.ROUND_UP);
	}

	String newsize = cursize.toString();
	return newsize;
    }
    
    /**
     * Creates a minidisk link for a user that allows that user to gain access to
     * another guests minidisk
     * Wraps  {@link live.smapi.SMAPI_Interface#imageDiskShare(String, String, String, String, String, String)}.
     *
     * @param sourceGuest The guest owning the disk to share.
     * @param sourceVaddr The virtual address of the disk relative to source guest.
     * @param destGuest The guest obtaining the link.
     * @param destVaddr The virtual address of the disk relative to destination guest.
     * @param accessMode PErmissions the linker will gave to read/write the disk.  Can be any valid CP access mode (i.e: RR, WR, MR,etc).
     * @throws ZVMTasksException An error occurred, just passes up SMAPIException
     */
    public void shareDisk(String sourceGuest, String sourceVaddr, String destGuest, String destVaddr, String accessMode)
	throws ZVMTasksException
    {
	//Create the directory entry for this disk.
	//Password not required for linking in directory.
	try
	{
	    smapi.imageDiskShare(sourceGuest, sourceVaddr, destGuest, destVaddr, accessMode, "");
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	//Now we need to make the static link.  To do this, we'll need the password.
	//String passwords = getMDiskPassword(sourceGuest, sourceVaddr, 6);
	
	/*
	 *  Giving up on dynamic linking for now... its a bit sketchy because we need to
	 *  use the password, if the disk in question does not have a password, it cannot
	 *  be linked to.
	 * We get an "ImageDiskNotAvailable" exception when we try synamic linking
	 * to a disk with no password in the directory.
	 *
	 * YES this exception is what is thrown when you try to link to a disk
	 * with no passwords defined... its dumb... I know, :)
	 *
	 **/
    }
    
    /**
     * Removes a minidisk link from a guest
     * Wraps  {@link live.smapi.SMAPI_Interface#imageDiskUnshare(String, String, String, String)}.
     *
     * @param sourceGuest The guest owning the disk being share.
     * @param sourceVaddr The virtual address of the disk relative to source guest.
     * @param destGuest The guest that has the link.
     * @param destVaddr The virtual address of the disk relative to destination guest.
     * @throws ZVMTasksException An error occurred, just passes up SMAPIException
     */
    public void unshareDisk(String sourceGuest, String sourceVaddr, String destGuest, String destVaddr)
	throws ZVMTasksException
    {
	//Remove the directory entry for this disk.
	try
	{
	    smapi.imageDiskUnshare(destGuest, destVaddr, sourceGuest, sourceVaddr);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    
   /** 
    * This function will clone an existing guest by copying all it's characteristics (except password)
    *  to a new guest
    * Wraps {@link live.smapi.SMAPI_Interface#imageCreate(String, String, String)}.
    *
    * @param existGuestName Name of guest to clone
    * @param cloneGuestName Name of clone
    * @param clonePassword Password of clone
    * @return {@link java.util.LinkedList} of minidisk addresses and the result of the disk operations
    * @throws ZVMTasksException An error occurred
    */
    public LinkedList cloneGuest(String existGuestName, String cloneGuestName, String clonePassword, boolean cloneDisks, boolean formatDisks)
	throws ZVMTasksException
    {    	
	LinkedList existGuestDirectory; // = new LinkedList();
	try
	{
	    //Make sure this prototype exists
	    ensurePrototypeExists(PROFILE_GENERAL);

	    //Create new image
	    smapi.imageCreate(cloneGuestName, PROFILE_GENERAL, clonePassword);

	    //Query the guest we want to clone
	    existGuestDirectory = smapi.imageQuery(existGuestName);

	}
	catch (SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}

	LinkedList aAndS; // = new LinkedList();
	
	// Try to copy the disks, if it fails, delete the guest completely 
	// and tell the user the cloning operation failed
	if (cloneDisks)
	{
	    try
	    {
		aAndS = cloneCopyMDisks(existGuestDirectory, cloneGuestName, clonePassword, existGuestName);
	    }
	    catch (SMAPIException e)
	    {
		try
		{
		    deleteImage(cloneGuestName);
		}
		catch (ZVMTasksException t) {}
		throw new ZVMTasksException(e);
	    }
	}
	
	//Again, try the creation operation, if it fails, delete the guest and tell the user the
	// operation failed
	else
	{
	    try
	    {
		aAndS = cloneCreateMDisks(existGuestDirectory, cloneGuestName, clonePassword, formatDisks);
	    }
	    catch(SMAPIException e)
	    {
		try
		{
		    deleteImage(cloneGuestName);
		}
		catch (ZVMTasksException t) {}
		throw new ZVMTasksException(e);
	    }
	}
	return aAndS;
    } 
    
    /** 
     * This function will go through and update and make the clone's directory the same as the 
     * existing guest's and will return a list of the minidisks for either creation or copying
     * Wraps {@link live.smapi.SMAPI_Interface#imageReplace(String, LinkedList)}.
     * 
     * @param existGuestDirectory Linked List of the existing Guest's directory statements
     * @param cloneGuestName name of the clone
     * @param clonePassword password for the clone
     * @return {@link java.util.LinkedList} of {@link live.dto.MDisk}s
     * @throws ZVMTasksException An error occurred, just passes up the SMAPIException
     */
    public LinkedList changeGuestsDirectory(LinkedList existGuestDirectory, String cloneGuestName, String clonePassword)
	throws ZVMTasksException
    {
	//Change name and password in existGuestDirectory
	String newUserLine="";
	StringTokenizer strTok = new StringTokenizer((String)existGuestDirectory.removeFirst());
	if (strTok.countTokens() < 6) throw new ZVMTasksException("changeMem: not enough tokens in USER line!");

	newUserLine = newUserLine + strTok.nextToken() + " "; //Keep 'USER'
	strTok.nextToken();//Eat guest name
	newUserLine = newUserLine + cloneGuestName + " "; //Replace guest's name
	strTok.nextToken();//Eat password
	newUserLine = newUserLine + clonePassword + " "; //Replace guest's password
	newUserLine = newUserLine + strTok.nextToken() + " "; //Keep initMem
	newUserLine = newUserLine + strTok.nextToken() + " "; //Keep maxMem
	newUserLine = newUserLine + strTok.nextToken(); //Keep permission classes

	existGuestDirectory.addFirst(newUserLine); 
	//Iterate through existing GuestDirectory
	ListIterator iter = existGuestDirectory.listIterator();
	LinkedList Disks =  new LinkedList();
	
	//Go through list
	while(iter.hasNext())
	{
	    //Tokenize each line in the user directory
	    StringTokenizer stringTok = new StringTokenizer((String)iter.next());
	    String firstWord = stringTok.nextToken();
	    
	    //See what kind of statement it is, if we can't use it, remove it
	    if(firstWord.equals("DEDICATE") | firstWord.equals("SPECIAL"))
		iter.remove();
	    
	    //if it's a minidisk statement, parse it out
	    else if(firstWord.equals("MDISK"))
	    {
		MDisk thisDisk = new MDisk();
		//remove it from the directory
		iter.remove();
		String addr = stringTok.nextToken();
		
		//There are already six minidisks defined in the IBMDFLT profile, so we don't want to recreate those
		if (!addr.equals("0190") && !addr.equals("019D") && !addr.equals("019E") && !addr.equals("0402") && !addr.equals("0401") && !addr.equals("0405"))
		{
		    //First get the virtual address
		    thisDisk.setVirtAddr(addr);	
		    
		    //Get the device type (3390)
		    String devType = stringTok.nextToken();
		    //3390 is the device type for DASD, which is specified in cylinders
		    if (devType.equals("3390"))
			thisDisk.setBlOrCyl("CYLINDERS");
		    else
			thisDisk.setBlOrCyl("BLK4096");
		    
		    //get the starting number for the minidisk
		    thisDisk.setStartLoc(stringTok.nextToken());

		    //get the size of the minidisk
		    String amt = stringTok.nextToken();
		    
		    //the size could be specified as the entire rest of the available space
		    if (amt.equals("END"))
		    {
			//Cylinders are one size
			if (thisDisk.getBlOrCyl().equals("CYLINDERS"))
			{
			    int amount = 1113 - Integer.parseInt(thisDisk.getStartLoc());
			   thisDisk.setSize(Integer.toString(amount));
			}
			//VDisks are another size
			else
			{
			    int amount = 4194296 - Integer.parseInt(thisDisk.getStartLoc());
			    thisDisk.setSize(Integer.toString(amount));
			}
		    }
		    //or the size is actually just a regular size
		    else
			thisDisk.setSize(amt);
		    
		    
		    //if it's not a TDisk or VDisk then the next token will be the volume ID that the DASD's coming from
		    if (stringTok.hasMoreTokens() && !(thisDisk.getStartLoc().equals("V-DISK") | thisDisk.getStartLoc().equals("T-DISK")))
		    {
			thisDisk.setVolumeID(stringTok.nextToken());
		    }
		    //if it IS a TDisk or VDisk the next token is access mode
		    else
		    {
			thisDisk.setMode(stringTok.nextToken());
			
			//Check for passwords
			if(stringTok.hasMoreTokens())
			{
			    thisDisk.setHasPW(true);
			    thisDisk.setReadPW(stringTok.nextToken());
			    if (stringTok.hasMoreTokens()) thisDisk.setWritePW(stringTok.nextToken());
			    if (stringTok.hasMoreTokens()) thisDisk.setMultiPW(stringTok.nextToken());
			}
		    }
		    
		    //For DASD only, access mode
		    if (stringTok.hasMoreTokens())
		    {
			thisDisk.setMode(stringTok.nextToken());
			
			//Check for passwords
			if(stringTok.hasMoreTokens())
			{
			    thisDisk.setHasPW(true);
			    thisDisk.setReadPW(stringTok.nextToken());
			    if (stringTok.hasMoreTokens()) thisDisk.setWritePW(stringTok.nextToken());
			    if (stringTok.hasMoreTokens()) thisDisk.setMultiPW(stringTok.nextToken());
			}
		    }
		}
		//Add it to our list of disks
		Disks.add(thisDisk);
	    }
	}
	
	//Lock our guest before we try a replace
	try
	{
	    smapi.imageLock(cloneGuestName);
	}
	catch(SMAPIException e) { }
	
	//Do the Image Replace with the new guest directory
	try
	{
	    smapi.imageReplace(cloneGuestName,  existGuestDirectory); //Will unlock image for us if successful.
	}
	catch(SMAPIException e)
	{
	    // Since we are not sure if the guest is locked or not, we do not
	    // care if the unlock call fails or not.  This is just a precaution to
	    // help ensure we do not lock the guest and leave it in a locked state.
	    try { smapi.imageUnlock(cloneGuestName); }
	    catch(SMAPIException f) { }
	    
	    e.printStackTrace();
	    throw new ZVMTasksException(e);
	}
	return Disks;
    }
   /**
    * Takes an existing Guest Directory, takes out DEDICATE and SPECIAL statements
    *  takes MDISK statements and creates a list of minidisks for the clone of the same size, type etc 
    * Wraps {@link live.smapi.SMAPI_Interface#imageDiskCreateDM(String, String, String, String, String, String, String, String, String, String)}.
    *
    * @param existGuestDirectory directory of the guest you're cloning
    * @param cloneGuestName name of the clone
    * @param clonePassword password for the clone
    * @param formatDisks whether to format the disks
    * @return {@link java.util.LinkedList} of {@link live.dto.MDisk}s
    * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
    * @throws SMAPIException A disk creation error occurred
    */
    public LinkedList cloneCreateMDisks(LinkedList existGuestDirectory, String cloneGuestName, String clonePassword, boolean formatDisks)
	throws ZVMTasksException, SMAPIException
    {
	LinkedList Disks = changeGuestsDirectory(existGuestDirectory, cloneGuestName, clonePassword);

	LinkedList addresses = new LinkedList();
	//Creating new disks
	
	//Determine formatting
	String format = "NONE";
	if (formatDisks)
	    format = "CMS";

	//Go through Disks and create all the new disks
	ListIterator iterd = Disks.listIterator();
	while(iterd.hasNext())
	{
	    MDisk thisDisk = (MDisk)iterd.next();
	    smapi.imageDiskCreateDM(cloneGuestName, thisDisk.getVirtualAddr(), 
			"AUTOG", thisDisk.getBlOrCyl(), 
			thisDisk.getSize(), thisDisk.getMode(), format, 
			thisDisk.getReadPW(), thisDisk.getWritePW(), 
			thisDisk.getMultiPW());
		addresses.add("0");
	    addresses.add(thisDisk.getVirtualAddr());
	}
	return addresses;
    }
   /**
    * Takes an existing Guest Directory, takes out DEDICATE and SPECIAL statements
    *  takes MDISK statements and copies each one of the disks
    * Wraps {@link live.smapi.SMAPI_Interface#imageDiskCopyDM(String, String, String, String, String, String, String, String, String)}.
    *  
    * @param existGuestDirectory directory of the guest you're cloning
    * @param cloneGuestName name of the clone
    * @param clonePassword clone's password
    * @param existGuestName name of the guest you are cloning
    * @return {@link java.util.LinkedList} of returncodes and addresses
    * @throws ZVMTasksException An error occurred, just passes up the SMAPIException
    * @throws SMAPIException An error occurred
    */
    public LinkedList cloneCopyMDisks(LinkedList existGuestDirectory, String cloneGuestName, String clonePassword, String existGuestName)
	throws ZVMTasksException, SMAPIException
    {
	
	LinkedList Disks = changeGuestsDirectory(existGuestDirectory, cloneGuestName, clonePassword);
	LinkedList addressesAndSuccesses = new LinkedList();
	//Creating new disks
	//Go through Disks and create all the new disks
	ListIterator iterd = Disks.listIterator();
	while(iterd.hasNext())
	{
	    MDisk thisDisk = (MDisk)iterd.next();
	    int i = smapi.imageDiskCopyDM(cloneGuestName, thisDisk.getVirtualAddr(),
	    				existGuestName,	thisDisk.getVirtualAddr(), 
					thisDisk.getAllocationType(), thisDisk.getMode(), 
					thisDisk.getReadPW(), thisDisk.getWritePW(), thisDisk.getMultiPW());
	    String inti = Integer.toString(i);
	    addressesAndSuccesses.add(inti);
	    addressesAndSuccesses.add(thisDisk.getVirtualAddr());
	}

	return addressesAndSuccesses;
    }
    
   /**
     * This function creates a new guest with certain specified characteristics 
     * Wraps {@link live.smapi.SMAPI_Interface#imageCreate(String, String, String)}.
     *
     * @param guestName Name of new guest
     * @param password Password
     * @param initMem Initial amount of memory the guest has
     * @param maxMem Maximum amount of memory the guest can have
     * @param prototypeName Prototype to use for this guest (usually we'll use our default, except for cloned guests)
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */    
    public void createGuest(String guestName, String password, String initMem, String maxMem, String prototypeName)
	throws ZVMTasksException
    {
	try
	{
	    //Make sure our profile exists
	    ensurePrototypeExists(prototypeName);

	    //Create guest
	    smapi.imageCreate(guestName, prototypeName, password);
	    
	    //Set guests memory
	    changeMemory(guestName,  initMem,  maxMem);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
    }
    
    /**
     * This function changes the initial and maximum allotted memory for a guest
     * Wraps {@link live.smapi.SMAPI_Interface#imageReplace(String, LinkedList)}
     *
     * @param guestName Name of guest to change memory of
     * @param newInitMem Initial amount of memory you want the guest to have
     * @param newMaxMem Maximum amount of memory you want the guest to be able to have
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception
     */   
    public void changeMemory(String guestName, String newInitMem, String newMaxMem)
	throws ZVMTasksException
    {
	LinkedList guestDef = null;
	
	try
	{
	    //Get guest
	    guestDef = smapi.imageQuery(guestName);

	    //Replace mem
	    String newUserLine="";
	    StringTokenizer strTok = new StringTokenizer((String)guestDef.removeFirst());
	    if (strTok.countTokens() < 3) throw new ZVMTasksException("changeMem: not enough tokens in USER line!");

	    newUserLine = newUserLine + strTok.nextToken() + " "; //Eat 'USER'
	    newUserLine = newUserLine + strTok.nextToken() + " "; //Eat guest's name
	    newUserLine = newUserLine + strTok.nextToken() + " "; //Eat guest's pasword
	    
	    //Some guests have NO more tokens in their user line!
	    if(strTok.countTokens() >= 3)
	    {
		strTok.nextToken(); //Eat initMem
		strTok.nextToken(); //Eat maxMem
		newUserLine = newUserLine + newInitMem + " " + newMaxMem + " ";
		newUserLine = newUserLine + strTok.nextToken(); //Eat permission classes
	    }
	    else
	    {
		newUserLine = newUserLine + newInitMem + " " + newMaxMem + " ";
	    }

	    guestDef.addFirst(newUserLine);
	    
	    //Lock guest for replacement
	    smapi.imageLock(guestName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	 
	// The reason we have a 2nd try/catch block here is because if we have an error
	// in the replacement its safe to unlock this guest because we KNOW that we
	// locked it.  We know this because if the lock call above had failed we
	// would never have gotten here.
	//
	try
	{
	    smapi.imageReplace(guestName,  guestDef); //Will unlock image for us if successful.
	}
	catch(SMAPIException e)
	{
	    // Since we are not sure if the guest is locked or not, we do not
	    // care if the unlock call fails or not.  This is just a precaution to
	    // help ensure we do not lock the guest and leave it in a locked state.
	    try { smapi.imageUnlock(guestName); }
	    catch(SMAPIException f) { }
	    
	    throw new ZVMTasksException(e);
	}
    }
    
   /**
     * This function makes sure that the given prototype exists, if it does not, it creates it
     *  (this will probably be used at server startup for our generic prototype)
     * Wraps {@link live.smapi.SMAPI_Interface#prototypeCreate(String, LinkedList)}.
     *
     * @param prototypeName Name of profile to look for 
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */  
    private void ensurePrototypeExists(String prototypeName)
	throws SMAPIException
    {
	LinkedList pLines = new LinkedList();
	if (prototypeName.equals(PROFILE_SUPER))
	{
	    pLines.add("USER LIVEP1 XXXXXXXX 1M 1M ABCDEFGHIJKLMNOPQRSTUVWXYZ");
	    pLines.add("INCLUDE IBMDFLT");
	    pLines.add("IPL CMS");
	}
	else  // prototypeName.equals(PROFILE_GENERAL)
	{
	    pLines.add("USER LIVEP0 XXXXXXXX 1M 1M G");
	    pLines.add("INCLUDE IBMDFLT");
	    pLines.add("IPL CMS");
	}
	
	try { smapi.prototypeCreate(prototypeName, pLines); }
	catch(PrototypeAlreadyExistsException e) { }
    }
    
    /**
     * Creates a new shared segment given the name of the segment, its size and a
     * list of guests you wish to eventually add to the segment.  This function
     * does not actually give those guests access to the segment, instead they are
     * used as a model so a correct and non-conflicting page range can be chosen when
     * creating the segment.
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageCreate(String, String, String, String)}
     *  & {@link live.zvmtasks.ZVMTasks#getPageRangeFromGuests(LinkedList, String)}.
     *
     *
     * @param segName Name of the segment to create.
     * @param segSizeInMB The size of the segment you wish to create in Megabytes.
     * @param guestsToAdd List of guests to use as model for determining page range.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public void sharedStorageCreate(String segName, String segSizeInMB, LinkedList guestsToAdd)
	throws ZVMTasksException
    {
	//SEG_BUILDER must be active in order to build a segment.
	try
	{
	    smapi.imageActivate(SEG_BUILDER);
	}
	catch(ImageAlreadyActiveException e)
	{
	    //If the guest is already active, do nothing.
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	try
	{
	    String pageRange =  getPageRangeFromGuests(guestsToAdd, segSizeInMB);
	    
	    smapi.sharedStorageCreate(SEG_BUILDER, segName, pageRange, "EN");
	    
	    //In order to ensure SEG_BUILDER will be able to delete the segment,
	    //we need to give him access to it.
	    smapi.sharedStorageAddAccess(SEG_BUILDER, segName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
    }
    
    /**
     * Given a list of guests, they will all be querried and a page range will
     * be returned that is above all of the guests address spaces and that is not
     * currently being used for any shared segment.  This range is appropriate for
     * use in creating a new shared segment that will be accessed by all guests in
     * the list.
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageFindPageRange(String, String)}
     * 
     * @param guests Guests to consider when choosing the page range.
     * @param segSizeInMB Size of the page range to return.
     * @return a string in the format nnnn-mmmm where nnnn and mmmm are both hexadecimal
     * page numbers.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    private String getPageRangeFromGuests(LinkedList guests, String segSizeInMB)
	throws ZVMTasksException
    {
	LinkedList ranges = new LinkedList();
	
	try
	{
	    ListIterator itr = guests.listIterator();
	    while(itr.hasNext())
	    {
		String curGuest = (String)itr.next();
		System.out.println("Getting pg rng for _" + curGuest + "_");
		String curRange = smapi.sharedStorageFindPageRange(segSizeInMB, curGuest);
		
		//Range looks like this:  2300-33FF
		//Must break it up.
		StringTokenizer strTok = new StringTokenizer(curRange, "-");
		String startPage = strTok.nextToken();
		String endPage = strTok.nextToken();
		
		SavedSegment seg = new SavedSegment("",startPage, endPage);
		
		//Now we need to store the values as integers for later calculation.
		//Even though we do not have a nuch of segments here, we'll use the
		//SavedSegment DTO because it allows us to store a page range.
		int intStartPage = Integer.decode( "0x" + startPage).intValue();
		int intEndPage = Integer.decode( "0x" + endPage).intValue();
		
		seg.setIntStartPage(intStartPage);
		seg.setIntEndPage(intEndPage);
		
		//Finally, add current seg to the list.
		System.out.println("Adding " + seg.getPageRange());
		ranges.add(seg);
	    }
	    
	    //Now we have a list of ranges and we need to find the one with the
	    //highest start value.
	    ListIterator rangeItr = ranges.listIterator();
	    SavedSegment highestSeg = (SavedSegment)rangeItr.next();
	    while(rangeItr.hasNext())
	    {
		SavedSegment curSeg = (SavedSegment)rangeItr.next();
		
		if(curSeg.getIntStartPage() > highestSeg.getIntStartPage())
		{
		    highestSeg = curSeg;
		}
	    }
	    
	    return highestSeg.getPageRange();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	
    }
    
    /**
     * Deletes a shared storage segment, given the name.
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageDelete(String, String)}. 
     *
     * @param segName Name of segment to delete.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public void sharedStorageDelete(String segName)
	throws ZVMTasksException
    {
	//SEG_BUILDER must be active in order to delete a segment.
	try
	{
	    smapi.imageActivate(SEG_BUILDER);
	}
	catch(ImageAlreadyActiveException e)
	{
	    //If the guest is already active, do nothing.
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	try
	{
	    smapi.sharedStorageDelete(SEG_BUILDER,  segName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	//If the delete was a asuccess, we'll remove SEG_BUILDER from our
	//namelist.
	try
	{
	    smapi.nameListRemove("LIVESEG_" + segName, SEG_BUILDER);
	}
	catch(SMAPIException e)
	{
	    //If it failed, no need to do anything special.
	}
    }
    
    /**
     * Returns a linkedlist of SavedSegment objects that correspond to
     * the shared segments that exist on the system
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageQuery()}.
     *
     * @return {@link java.util.LinkedList} of {@link live.dto.SavedSegment}s.
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public LinkedList sharedStorageQuery()
	throws ZVMTasksException
    {
	LinkedList segStrings = null;
	
	try
	{
	    segStrings = smapi.sharedStorageQuery();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	LinkedList segments = new LinkedList();
	SavedSegment curSeg;
	ListIterator itr = segStrings.listIterator();
	while(itr.hasNext())
	{
	    StringTokenizer strTok = new StringTokenizer((String)itr.next());
	    curSeg = new SavedSegment(strTok.nextToken(), strTok.nextToken(),  strTok.nextToken());
	    segments.add(curSeg);
	}
	
	return segments;
    }
    
    /* This function will add access to a shared storage segment
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageAddAccess(String, String)}.
     *
     * @param guestName Name of the guest to grant access
     * @param segmentName Name of the saved segment for which to grant access
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public void sharedStorageAccessAdd(String guestName, String segmentName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.sharedStorageAddAccess(guestName, segmentName);
	    smapi.nameListAdd("LIVESEG_" + segmentName, guestName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	System.out.print(".");
    }
    
    /**
     * Revokes a guest's access to a saved segment
     * Wraps {@link live.smapi.SMAPI_Interface#sharedStorageRemoveAccess(String, String)}.
     *
     * @param guestName Guest to revoke access for 
     * @param segmentName Saved segment to revoke access from
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public void sharedStorageAccessRemove(String guestName, String segmentName)
	throws ZVMTasksException
    {
	try
	{
	    smapi.sharedStorageRemoveAccess(guestName, segmentName);
	    smapi.nameListRemove("LIVESEG_" + segmentName, guestName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	System.out.print(".");
    }
    
    /**
     *
     *
     */
    //TODO: Javadoc this!
    public boolean sharedStorageAccessQuery(String guestName, String segmentName)
	throws ZVMTasksException
    {
	boolean access = false;
	try
	{
	    access = smapi.sharedStorageQueryAccess(guestName,  segmentName);
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	System.out.print(".");
	return access; //If event of error, we never get here.
    }
    
    /**
     * Gets the name list with the access information for a saved segment
     * Wraps {@link live.smapi.SMAPI_Interface#nameListQuery(String)}.
     * 
     * @param segmentName Name of the saved segment to query
     * @throws ZVMTasksException An error occurred.  Just passes up the SMAPI exception.
     */
    public LinkedList sharedStorageAccessQueryNameList(String segmentName)
	throws ZVMTasksException
    {
	LinkedList guestsInList = null;
	try
	{
	    guestsInList = smapi.nameListQuery("LIVESEG_" + segmentName);
	}
	catch(ListNotFoundException e)
	{
	    //If there is not list, return an emtpy linked list
	    // We do this so as not to cause an error when listing a shared 
	    // segment that has no access list.
	    guestsInList = new LinkedList();
	}
	catch(SMAPIException e)
	{
	    throw new ZVMTasksException(e);
	}
	
	return guestsInList; //In event of error, we never get here.
    }
    
}
