/*
 * ZVMTasks.java Created on June 14, 2005, 12:41 PM
 */

package live.threads;

import live.zvmtasks.ZVMTasks;
import live.zvmtasks.ZVMTasksException;

import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Collection;
import live.dto.Guest;
import live.dto.Network;

/**
 * This library serves as a wrapper library for ZVMTasks, wrapping each in a
 * thread.
 * 
 * @author Evan Driscoll
 */
public class ThreadedZVMTasks
{
	/**
	 * Retrieves some of the info from a guest's directory entry. Wraps
	 * {@link live.zvmtasks.ZVMTasks#queryGuest(String)}.
	 */
	static public class QueryGuestInfo extends ThreadedUserAction
	{
		private Guest info;
		private String guestName;

		/**
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 * @param guestNameToQuery The guest's directory entry to return
		 */
		public QueryGuestInfo(ThreadInitInfo initInfo, String guestNameToQuery)
		{
			init(initInfo);
			guestName = guestNameToQuery;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#queryGuest(String)}.
		 * @throws ZVMTasksException passes up the SMAPI exception
		 */
		protected void runImpl() throws ZVMTasksException
		{
			info = tasks.queryGuest(guestName);
		}
		
		/**
		 * Returns the description of this task.
		 * @return String description
		 */
		public String description()
		{
			return "Querying information about user " + guestName;
		}

		/**
		 * Returns a {@link live.dto.Guest} object with the fields filled in,
		 * except for the group name. Or, it returns an exception thrown
		 * by {@link live.zvmtasks.ZVMTasks#queryGuest(String)} or null
		 * if that call hasn't completed.
		 * 
		 * @return A filled out <tt>Group</tt>, <tt>Exception</tt>, or <tt>null</tt>
		 */
		public Object rawResult()
		{
			if(info != null)
				return info;
			else
				return getException();
		}
	}
	
	/**
	 * Tests wether this user can authenticate to the SMAPI server or not.
	 */
	static public class Authenticate extends ThreadedUserAction
	{
		private Boolean success;
		private String user; //, pass;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#authenticate()}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public Authenticate(ThreadInitInfo initInfo)
		{
			init(initInfo);
			user = initInfo.getConnectionInfo().getUserName();
			//pass = initInfo.getConnectionInfo().getPassword();
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#authenticate()}
		 * @throws ZVMTasksException Just a SMAPI Exception called up
		 */
		protected void runImpl() throws ZVMTasksException
		{
			success = Boolean.valueOf(tasks.authenticate());
		}

		/**
		 * Description of this task
		 * @return String description
		 */
		public String description()
		{
			return "Authenticating with " + user;
		}

		/**
		 * Returns a {@link java.lang.Boolean}object. The value is as follows:
		 * 1. null: The {@link live.zvmtasks.ZVMTasks#authenticate()}call has
		 * not completed 2. Boolean(true): the authentication was successful 3.
		 * Boolean(false): the authentication was unsuccessful
		 * 
		 * @return A {@link java.lang.Boolean}indicating whether the
		 *         authentication was successful
		 */
		public Object rawResult()
		{
			if(success != null)
				return success;
			else
				return getException();
		}
	}

	/**
	 * Returns a linkedlist of Networks. Each Network is populated with its name
	 * and network type (Guest LAN or VSwitch)
	 */
	static public class QueryAllNetworks extends ThreadedUserAction
	{
		private LinkedList networkList;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#queryAllNetworks()}. The parameter
		 * initInfo MUST be non-null or this will throw a
		 * {@link NullPointerException}.
		 * 
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 */
		public QueryAllNetworks(ThreadInitInfo initInfo)
		{
			init(initInfo);
		}
		
		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#queryAllNetworks()}
		 */
		protected void runImpl() throws ZVMTasksException
		{
			networkList = tasks.queryAllNetworks();
		}

		/**
		 * Description of this task.
		 * @return String description
		 */
		public String description()
		{
			return "Querying all networks";
		}

		/**
		 * Returns the {@link java.lang.LinkedList} that was returned from
		 * {@link live.zvmtasks.ZVMTasks#queryAllNetworks()} if the call was
		 * successful, the {@link live.zvmtasks.ZVMTasksException} that was 
		 * thrown if not, or <tt>null</tt> if the call has not been completed.
		 * 
		 * @return The {@link LinkedList}returned from
		 *         {@link live.zvmtasks.ZVMTasks#queryAllNetworks()} if
		 *         successful, an Exception, or null
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return networkList;
			else
				return getException();
		}
	}
	
	
	/**
	 * Returns a linkedlist of Networks. Each Network is populated with its name
	 * and network type (Guest LAN or VSwitch)
	 */
	static public class QueryNetworkMembers extends ThreadedUserAction
	{
		private Network network;
		private String netName;
		
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#getNetworkMembers(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 * @param netName The network name to query
		 */
		public QueryNetworkMembers(ThreadInitInfo initInfo, String netName)
		{
			this.netName = netName;
			init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#getNetworkMembers(String)}
		 */
		protected void runImpl() throws ZVMTasksException
		{
			network = tasks.getNetworkMembers(netName);
		}

		/**
		 * Returns a description of this task.
		 * @return String description.
		 */
		public String description()
		{
			return "Getting members of network " + netName;
		}

		/**
		 * Returns the {@link live.dto.Network} that was returned from
		 * {@link live.zvmtasks.ZVMTasks#getNetworkMembers(String)} if the call was
		 * successful, the ZVMTasksException that was thrown if not, or null if
		 * the call has not been completed.
		 * 
		 * @return A <tt>Network</tt>, exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return network;
			else
				return getException();
		}
	}

	/**
	 * Returns a linkedlist of Strings. Each String is the name of a guest on
	 * the system.
	 */
	static public class QueryAllGuestNames extends ThreadedUserAction
	{
		private LinkedList guestList;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#queryAllGuestNames()}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public QueryAllGuestNames(ThreadInitInfo initInfo)
		{
			init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#queryAllGuestNames()}
		 */
		protected void runImpl() throws ZVMTasksException
		{
			guestList = tasks.queryAllGuestNames();
		}

		/**
		 * Returns a description of the task.
		 * @return String description
		 */
		public String description()
		{
			return "Querying all guests";
		}

		/**
		 * Returns the {@link java.lang.LinkedList} that was returned from
		 * {@link live.zvmtasks.ZVMTasks#queryAllGuestNames()} if the call was
		 * successful, the ZVMTasks exception that was thrown if not, or null if
		 * the call has not been completed.
		 * 
		 * @return The {@link LinkedList}returned from
		 *         {@link live.zvmtasks.ZVMTasks#queryAllGuestNames()}, or an
		 *         Exception, or null
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return guestList;
			else
				return getException();
		}
	}

	
	////// The following threaded actions and the underlying tasks methods were 
	////// rendered unneeded when we dropped guest LAN support. As of this writing, 
	////// they still work as well as the SMAPI function works, but we don't use 
	////// it so commented it out. If you uncomment it, you'll have to re-terminate 
	////// a few comments.
	
	/*
	 * Creates a new Guest LAN and attaches a guest to it.
	 * /
	static public class GuestLanCreateConnect extends ThreadedUserAction
	{
		private String lanName, memberName, memberVaddr;

		/ **
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#guestLANCreateConnect(String, String, String)}.
		 * The parameter initInfo MUST be non-null or this will throw a
		 * {@link NullPointerException}. Also, initInfo.getConnectionInfo()
		 * must be non-null or this will throw a {@link NullPointerException}
		 * when the thread is started.
		 * 
		 * <code>initInfo.getConnectionInfo()</code> should return the object
		 * containing the server's address and port, and the username and
		 * password to try to authenticate with.
		 * <code>initInfo.getActiveThreadsInfo()</code> should return the
		 * session object containing the in progress and complete lists.
		 * 
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 * /
		public GuestLanCreateConnect(ThreadInitInfo initInfo, String lanName,
				String memberName, String memberVaddr)
		{
			init(initInfo);
			this.lanName = lanName;
			this.memberName = memberName;
			this.memberVaddr = memberVaddr;
		}

		protected void runImpl() throws ZVMTasksException
		{
			tasks.guestLANCreateConnect(lanName, memberName, memberVaddr);
		}

		public String description()
		{
			return "Creating/Connecting guest LAN " + lanName + " with member "
					+ memberName;
		}

		/ **
		 * Returns null, as
		 * {@link live.zvmtasks.ZVMTasks#guestLANCreateConnect(String,String,String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#guestLANCreateConnect(String,String,String)}
		 * /
		public Object rawResult()
		{
			if(exception == null)
				return null;
			else
				return exception;
		}
	}*/

	/*
	 * This function will disconnect a guest from a LAN.
	 * /
	static public class GuestLanDeleteDisconnect extends ThreadedUserAction
	{
		private String lanName, memberName, vaddr;

		/ **
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#guestLANDeleteDisconnect(String, String, String)}.
		 * The parameter initInfo MUST be non-null or this will throw a
		 * {@link NullPointerException}. Also, initInfo.getConnectionInfo()
		 * must be non-null or this will throw a {@link NullPointerException}
		 * when the thread is started.
		 * 
		 * <code>initInfo.getConnectionInfo()</code> should return the object
		 * containing the server's address and port, and the username and
		 * password to try to authenticate with.
		 * <code>initInfo.getActiveThreadsInfo()</code> should return the
		 * session object containing the in progress and complete lists.
		 * 
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 * /
		public GuestLanDeleteDisconnect(ThreadInitInfo initInfo,
				String lanName, String guestName, String vaddr)
		{
			init(initInfo);
			this.lanName = lanName;
			this.memberName = guestName;
			this.vaddr = vaddr;
		}

		protected void runImpl() throws ZVMTasksException
		{
			tasks.guestLANDeleteDisconnect(lanName, memberName, vaddr);
		}

		public String description()
		{
			return "Removing " + memberName + " from " + lanName;
		}

		/ **
		 * Returns null, as
		 * {@link live.zvmtasks.ZVMTasks#guestLANDeleteDisconnect(String,String,String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#guestLANDeleteDisconnect(String, String,String)}
		 * /
		public Object rawResult()
		{
			if(exception == null)
				return null;
			else
				return exception;
		}
	}*/
	
	/**
	 * This class finds the free address from several different guests
	 * and finds the lowest span of 3 addresses that are free in all of them.
	 * Wraps {@link live.zvmtasks.ZVMTasks#getCommonAddress(Collection)}.
	 */
	static public class FindCommonFreeAddress extends ThreadedUserAction
	{
		private int numberDone = 0;
		private Collection memberNames;
		private String hexAddress = null;
		
		private class ProgressListener 
			implements ZVMTasks.getCommonAddressProgressListener
		{
		       /**
			* Increments the number done
			*/
			public void anotherComplete()
			{
				++numberDone;
			}
		}
		
		ProgressListener listener = new ProgressListener();
		
		/**
		 * Sets up an action that will get list of free addresses for all the memberNames
		 * then compare them to get a commmon free address; 
		 * Wraps {@link live.zvmtasks#getCommonAddress(Collection, ProgressListener)}
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param memberNames A linked list of Strings that are names of
		 * 	the guests you want to find a common free address span among. 
		 */
		public FindCommonFreeAddress(ThreadInitInfo initInfo, Collection memberNames)
		{
			init(initInfo);
			this.memberNames = (memberNames == null) ? new LinkedList() : memberNames;
		}
		
		/** Returns the object being operated on so that it can be chained,
		 *  for instance a.addSource().addSource(), or b = a.addSource(p) will
		 *  add p to a, then make b point to a.
		 *
		 *  @param guestName
		 *  @return FindCommonFreeAddress The FindCommonFreeAddress method will 
		 *   be called and the name appended to the list of strings 
		 */
		public FindCommonFreeAddress addSource(String guestName)
		{
			memberNames.add(guestName);
			return this;
		}
		

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#getCommonAddress(Collection)}
		 */
		public void runImpl() throws ZVMTasksException
		{
		    hexAddress = tasks.getCommonAddress(memberNames, listener);
		}

		/**
		 * Returns a description of this task
		 * @return String description
		 */
		public String description()
		{
			String ret = "Finding common address";
			return ret;
		}
		
		/**
		 * Returns the resultString for the task, concatenated with 
		 *  how many queries have been done so far.
		 *  
		 * @return String result of the functions
		 */
		public String result()
		{
			if(getStatus() == ThreadedUserAction.IN_PROGRESS)
			{
				String ret = "In progress...";
				
				if(numberDone!=memberNames.size())
				{
					ret += " (" + numberDone + " of " + memberNames.size();
					ret += " queries done)";
				}
				else
				{
					ret += " (searching for common address)";
				}
				
				return ret;
			}
			else
				return super.result();
		}

		/**
		 * Returns a String with the four-digit hex address of the lowest
		 * free address, an Exception if there was an error or no common
		 * address can be found, or null if the call hasn't been completed.
		 * 
		 * @return A <tt>String</tt> with the common address in hex, exception, or null
		 */
		public Object rawResult()
		{
			if(hexAddress != null)
				return hexAddress;
			else
				return getException();
		}
	}
	
	/**
	 * Wraps 
	 * {@link live.zvmtasks.ZVMTasks#virtualSwitchCreateConnect(String, String, String, String, String)}.
	 * Connects a guest to a VSwitch, creating it if it doesn't exist.
	 * 
	 * Reccomendation: don't use this to create networks, because it causes 
	 * problems. In order for this to work properly when creating a network,
	 * at least one guest must be logged on. The static calls (the ones to
	 * Dirmaint) don't actually create the VSwitch, only the dynamic ones do.
	 * So if no guests are logged on, the VSwitch isn't created because the
	 * dynamic calls fail, then when a guest DOES log on the VSwitch doesn't
	 * exist for it to connect to. Also, no cache update is done from this.
	 */
	static public class VSwitchCreateConnect extends ThreadedUserAction
	{
		private String netName, memberName;
		private Object memberVaddr;
		private String portname, raddr;
		
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#virtualSwitchCreateConnect(String, String, String, String, String)}.

 		 * @param initInfo The ThreadInitInfo object
		 * @param netName The name of the VSwitch to create
		 * @param portname The name of the port to connect to or null
		 * @param raddr The address of a real OSA device to connect to, or null
		 * @param memberName The name of a member to connect to
		 * @param memberVaddr The FindCommonFreeAddress object
		 * 		that will return the virtual address from rawResult()
		 */
		public VSwitchCreateConnect(ThreadInitInfo initInfo, String netName,
				String portname, String raddr, String memberName,
				FindCommonFreeAddress memberVaddr)
		{
			init(initInfo);
			this.netName = netName;
			this.memberName = memberName;
			this.memberVaddr = memberVaddr;
			this.portname = portname;
			this.raddr = raddr;
		}
		
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#virtualSwitchCreateConnect(String, String, String, String, String)}.

 		 * @param initInfo The ThreadInitInfo object
		 * @param netName The name of the VSwitch to create
		 * @param portname The name of the port to connect to or null
		 * @param raddr The address of a real OSA device to connect to, or null
		 * @param memberName The name of a member to connect to
		 * @param memberVaddr The virtual address (in hex, no more than 4 digits)
		 * 		to connect the virtual NIC at
		 */
		public VSwitchCreateConnect(ThreadInitInfo initInfo, String netName,
				String portname, String raddr, String memberName,
				String memberVaddr)
		{
			init(initInfo);
			this.netName = netName;
			this.memberName = memberName;
			this.memberVaddr = memberVaddr;
			this.portname = portname;
			this.raddr = raddr;
		}
		
		/**
		 * Finds the lowest common free address amongst a collection of guests
		 * 
		 * @return String 4-digit hexadecimal virtual address
		 */
		private String getVirtualAddress()
		{
			System.out.println("In getVirtualAddress()");
			if(memberVaddr instanceof String)
			{
				System.out.println("The address was specified directly");
				return (String) memberVaddr;
			}
			
			if(memberVaddr instanceof ThreadedZVMTasks.FindCommonFreeAddress)
			{
				String addy = (String) ((ThreadedZVMTasks.FindCommonFreeAddress)memberVaddr).rawResult();
				System.out.println("The address was specified indirectly: " + addy);
				
				return addy; 
			}
			
			System.out.println("Address not specified. " + memberVaddr);
			return null;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#virtualSwitchCreateConnect(String, String, String, String, String)}
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.virtualSwitchCreateConnect(netName, memberName,
					getVirtualAddress(), portname, raddr);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Adding " + memberName + " to network " + netName +
			" at virtual address " + getVirtualAddress();
		}

		/**
		 * Returns null or an exception
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#guestLANCreateConnect(String,String,String)}
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return null;
			else
				return getException();
		}
	}
	/**
	 * This action will disconnect a guest from a LAN. Wraps
	 * {@link live.zvmtasks.ZVMTasks#virtualSwitchDetatch(String, String)}.
	 * (The address of the virtual NIC on the guest is detected automagically.)
	 * 
	 * This calls a custom SMAPI exec rather than detach/delete because
	 * that one has even worse problems than create/connect. If there were
	 * 5 guests with directory entries to attach to a VSwitch but only one
	 * was logged on, detaching that guest would delete the VSwitch.
	 */
	static public class VSwitchDetatch extends ThreadedUserAction
	{
		private String netName, guestName;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#virtualSwitchDetatch(String, String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param netName The VSwitch name to detach the guest from
		 * @param guestName The guest name to detach
		 */
		public VSwitchDetatch(ThreadInitInfo initInfo,
				String netName, String guestName)
		{
			init(initInfo);
			this.netName = netName;
			this.guestName= guestName;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#virtualSwitchDetatch(String, String)}.
		 */ 
		protected void runImpl() throws ZVMTasksException
		{
			tasks.virtualSwitchDetatch(netName, guestName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Removing " + guestName + " from " + netName;
		}

		/**
		 * Returns null or an exception
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#virtualSwitchDetatch(String,String)}
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return null;
			else
				return getException();
		}
	}
	
	/**
	 * This actually deletes a VSwitch. (It doesn't detach anyone, so
	 * be sure to call VSwitchDetach on each of the members beforehand.)
	 * Wraps
	 * {@link live.zvmtasks.ZVMTasks#virtualSwitchDelete(String)}.
	 */
	static public class VSwitchDelete extends ThreadedUserAction
	{
		private String netName;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#virtualSwitchDelete(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param netName The name of the VSwitch to delete 
		 */
		public VSwitchDelete(ThreadInitInfo initInfo, String netName)
		{
			init(initInfo);
			this.netName = netName;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#virtualSwitchDelete(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.virtualSwitchDelete(netName);
			getCache().removeNetwork(netName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Removing VSwitch " + netName;
		}

		/**
		 * Returns null or an exception
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#virtualSwitchDetatch(String,String)}
		 */
		public Object rawResult()
		{
			if(getException() == null)
				return null;
			else
				return getException();
		}
	}
	
	/// We don't do GuestLANs, real men use VSwitches
	/*
	static public class GuestLANCreate extends ThreadedUserAction
	{
		private String lanName;
		
		/ **
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#guestLANCreate(String)}.
		 * The parameter initInfo MUST be non-null or this will throw a
		 * {@link NullPointerException}. Also, initInfo.getConnectionInfo()
		 * must be non-null or this will throw a {@link NullPointerException}
		 * when the thread is started.
		 * 
		 * <code>initInfo.getConnectionInfo()</code> should return the object
		 * containing the server's address and port, and the username and
		 * password to try to authenticate with.
		 * <code>initInfo.getActiveThreadsInfo()</code> should return the
		 * session object containing the in progress and complete lists.
		 * 
		 * @param initInfo The ThreadInitInfo object that contains the
		 *            information explained above
		 * /
		public GuestLANCreate (ThreadInitInfo initInfo, String lanName)
		{
			init(initInfo);
			this.lanName = lanName;
		}

		protected void runImpl() throws ZVMTasksException
		{
			tasks.guestLANCreate(lanName);
		}

		public String description()
		{
			return "Creating LAN " + lanName;
		}

		/ **
		 * Returns null, as
		 * {@link live.zvmtasks.ZVMTasks#guestLANCreate(String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#guestLANCreate(String)}
		 * /
		public Object rawResult()
		{
			return exception;
		}
	}*/

	/**
	 * This creates a VSwitch without connecting anyone. (See
	 * {@link live.threads.ThreadedZVMTasks.VSwitchCreateConnect}
	 * for a description of why you shouldn't use that.) Wraps
	 * {@link live.zvmtasks.ZVMTasks#virtualSwitchCreate(String, String, String)}.
	 */
	static public class CreateVSwitch extends ThreadedUserAction
	{
		private final String portName;
		private final String rAddress;
		private final String netName;
		
		/**
		 * Creates an action that will call {@link live.zvmtasks.ZVMTasks#virtualSwitchCreate(String, String, String)}
		 * @param initInfo The ThreadInitInfo
		 * @param netName The name of the VSwitch you're creating
		 * @param portName The port name to connect the VSwitch to, or null
		 * @param realAddress The address of a real OSA device to connect to, or null
		 */
		public CreateVSwitch(ThreadInitInfo initInfo,
				String netName, String portName, String realAddress)
		{
			init(initInfo);
			
			this.netName = netName;
			this.portName = portName;
			this.rAddress = realAddress;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#virtualSwitchCreate(String, String, String)}
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.virtualSwitchCreate(netName, portName, rAddress);
			getCache().addNetwork(new Network(netName, Network.NETWORK_VSWITCH));
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Creating network " + netName;
		}

		/**
		 * Returns null or an exception
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This gets the groups that are on the server and their members.
	 * Wraps {@link live.zvmtasks.ZVMTasks#getGroups()}. 
	 */
	static public class GetGroups extends ThreadedUserAction
	{
		private LinkedList groups = null;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#getGroups()}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public GetGroups(ThreadInitInfo initInfo)
		{
			init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#getGroups()}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			groups = tasks.getGroups();
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Getting names of guest groups.";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.Group}s,
		 * an exception thrown by <tt>getGroups</tt>, or null if that call
		 * hasn't completed. 
		 * 
		 * @return A list of <tt>Group</tt> DTOs, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
			if(groups != null)
				return groups;
			else
				return getException();
		}
	}

	/**
	 * Gets a list of names of guests that are active. Wraps 
	 * {@link live.zvmtasks.ZVMTasks#activeImagesQuery()}. 
	 */
	static public class ActiveImagesQuery extends ThreadedUserAction
	{
		private LinkedList images;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#activeImagesQuery()}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public ActiveImagesQuery(ThreadInitInfo initInfo)
		{
			init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#activeImagesQuery()}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			images = tasks.activeImagesQuery();
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Querying active images.";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of Strings that are names
		 * of guests that are currently active, an exception thrown by
		 * <tt>activeImagesQuery</tt>, or <tt>null</tt>
		 * 
		 * @return A list of names of guests, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
			if(images != null)
				return images;
			else
				return getException();
		}
	}
	
	/**
	 * This activates a specified guest. Wraps
	 * {@link live.zvmtasks.ZVMTasks#imageActivate(String)}.
	 */
	static public class ImageActivate extends ThreadedUserAction
	{
		//private LinkedList images;
		private String imageName;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#imageActivate(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param imageName Guest to activate
		 */
		public ImageActivate(ThreadInitInfo initInfo, String imageName)
		{
		    this.imageName = imageName;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#imageActivate(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.imageActivate(imageName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
		    return "Activating image  " + imageName;
		}

		/**
		 * Returns null, as {@link live.zvmtasks.ZVMTasks#imageActivate(String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#imageActivate(String)}
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * Deactivates a specified guest. Wraps 
	 * {@link live.zvmtasks.ZVMTasks#imageDeactivate(String)}
	 */
	static public class ImageDeactivate extends ThreadedUserAction
	{
		//private LinkedList images;
		private String imageName;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#imageDeactivate(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param imageName Name of guest to deactivate
		 */
		public ImageDeactivate(ThreadInitInfo initInfo, String imageName)
		{
		    this.imageName = imageName;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#imageDeactivate(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.imageDeactivate(imageName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
		    return "Deactivating image " + imageName;
		}

		/**
		 * Returns null, as {@link live.zvmtasks.ZVMTasks#imageDeactivate(String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#imageDeactivate(String)}
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This moves a guest from one group to another. Wraps
	 * {@link live.zvmtasks.ZVMTasks#setGuestsGroup(String, String, String)}.
	 */
	static public class SetGuestToGroup extends ThreadedUserAction
	{
		private String newGroupName, oldGroupName, guestName;

		/**
		 * Creates an action that will call 
		 * {@link live.zvmtasks.ZVMTasks#setGuestsGroup(String, String, String)}.
		 *
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName The name of the guest to move
		 * @param oldGroupName The name of the group to remove <tt>guestName</tt>
		 * 		from, or null
		 * @param newGroupName The name of the group to add <tt>guestName</tt>
		 * 		to, or null
		 */
		public SetGuestToGroup(ThreadInitInfo initInfo, String guestName,
				String oldGroupName, String newGroupName)
		{
			init(initInfo);

			this.guestName = guestName;
			this.oldGroupName = oldGroupName;
			this.newGroupName = newGroupName;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#setGuestsGroup(String, String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.setGuestsGroup(guestName, oldGroupName,
					newGroupName);
			
			getCache().changeGuestsGroup(guestName, newGroupName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Moving " + guestName + " to group "
					+ newGroupName + " (from " + oldGroupName  + ")";
		}

		/**
		 * Returns null or an exception
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}

	
	/**
	 * This creates a guest. Wraps 
	 * {@link live.zvmtasks.ZVMTasks#createGuest(String, String, String, String, String)}.
	 */
	static public class CreateGuest extends ThreadedUserAction
	{
		private String name, pass, profile, defMem, maxMem;
		
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#createGuest(String, String, String, String, String)}. 
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param name The name of the new guest
		 * @param pass The password the new guest should initially have
		 * @param defMem The default amount of memory ("storage") the guest should
		 * 		have when activated
		 * @param maxMem The maximum amount of memory the guest should be able to set
		 * @param profile The name of the profile to base the guest off of 
		 */
		public CreateGuest(ThreadInitInfo initInfo, String name, 
				String pass, String defMem, String maxMem, String profile)
		{
			init(initInfo);
		    this.name = name;
		    this.pass = pass;
		    this.profile = profile;
		    this.maxMem = maxMem;
		    this.defMem = defMem;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#createGuest(String, String, String, String, String)}. 
		 */
		protected void runImpl() throws ZVMTasksException
		{
		    tasks.createGuest(name, pass, defMem, maxMem, profile);
		    getCache().addGuest(name);
		}

		/**
		 * Returns a description of the task
		 * @return String
		 */
		public String description()
		{
			return "Creating guest " + name;
		}

		/**
		 * Returns null, as <tt>createGuest</tt>
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or <tt>createGuest</tt>
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This deletes a guest. Wraps
	 * {@link live.zvmtasks.ZVMTasks#deleteImage(String)}.
	 */
	static public class DeleteGuest extends ThreadedUserAction
	{
		private String guestName;

		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#deleteImage(String)}. 
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName The name of the guest to delete
		 */
		public DeleteGuest(ThreadInitInfo initInfo, String guestName)
		{
		    init(initInfo);
		    this.guestName = guestName;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#deleteImage(String)}. 
		 */
		protected void runImpl() throws ZVMTasksException
		{
		    tasks.deleteImage(guestName);
		    getCache().removeGuest(guestName);
		}

		/**
		 * Returns a description of the task
		 * @return String
		 */
		public String description()
		{
			return "Deleting guest " + guestName;
		}

		/**
		 * Returns null, as {@link live.zvmtasks.ZVMTasks#deleteImage(String)}
		 * is void, unless the call was made and terminated by an exception, in
		 * which case this returns the ZVMTasksException that was thrown.
		 * 
		 * @return null or an exception thrown by
		 *         {@link live.zvmtasks.ZVMTasks#deleteImage(String)}
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This changes a guest's password. Wraps
	 * {@link live.zvmtasks.ZVMTasks#setGuestPassword(String, String)}.
	 */
	static public class ChangePassword extends ThreadedUserAction
	{
		private final String guestName;
		private final String newPass;
		
		/**
		 * Creates an action that will call {@link live.zvmtasks.ZVMTasks#setGuestPassword(String, String)}.
		 *
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName The name of the guest to change the password of
		 * @param newPass The guest's new password
		 */
		public ChangePassword(ThreadInitInfo initInfo,
				String guestName, String newPass)
		{
			init(initInfo);
			
			this.guestName = guestName;
			this.newPass = newPass;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#setGuestPassword(String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.setGuestPassword(guestName, newPass);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Changing " + guestName + "\'s password";
		}

		/**
		 * Returns <tt>null</tt> or an exception
		 * @return object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This changes a guest's memory. Wraps
	 * {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
	 */
	static public class ChangeMemory extends ThreadedUserAction
	{
		private final String guestName;
		private final String initMem;
		private final String maxMem;
		
		/**
		 * Creates an action that will call {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
		 *
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName The name of the guest
		 * @param initMem The initial memory given to a guest
		 * @param maxMem The maximum memory available to a guest
		 */
		public ChangeMemory(ThreadInitInfo initInfo,
				String guestName, String initMem, String maxMem)
		{
			init(initInfo);
			
			this.guestName = guestName;
			this.initMem = initMem;
			this.maxMem = maxMem;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.changeMemory(guestName, initMem, maxMem);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Changing " + guestName + "\'s memory";
		}

		/**
		 * Returns <tt>null</tt> or an exception
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * TODO
	 * This changes a guest's memory. Wraps
	 * {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
	 */
	static public class CloneGuest extends ThreadedUserAction
	{
		private final String existGuestName, cloneGuestName, clonePassword;
		private final boolean copyDisks, formatDisks;
		private LinkedList AddressAndSuccess;
		
		/**
		 * Creates an action that calls {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
		 *
		 * @param initInfo The ThreadInitInfo object
		 * @param existGuestName The name of the guest to clone
		 * @param cloneGuestName The name of the new (clone) guest
		 * @param clonePassword The clone's password
		 */
		public CloneGuest(ThreadInitInfo initInfo,
				String existGuestName, String cloneGuestName, 
				String clonePassword, boolean copyDisks, boolean formatDisks)
		{
			init(initInfo);
			
			this.existGuestName = existGuestName;
			this.cloneGuestName = cloneGuestName;
			this.clonePassword = clonePassword;
			this.copyDisks = copyDisks;
			this.formatDisks = formatDisks;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#changeMemory(String, String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			AddressAndSuccess = tasks.cloneGuest(existGuestName, cloneGuestName, clonePassword, copyDisks, formatDisks);
			getCache().addGuest(cloneGuestName);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
		    String retstring = "Creating " + cloneGuestName + " from " + existGuestName + "    Copy disks? " + copyDisks;
		    return retstring;
		}
		
		/**
		 * Return  the resultString concatenated with disk operation information
		 * @return String
		 */
		public String result()
		{
			if(getStatus() == ThreadedUserAction.COMPLETE)
			{
			    String retstring = " ";
		    
			    ListIterator iter = AddressAndSuccess.listIterator();
			    while(iter.hasNext())
			    {
				String successfail = (String)iter.next();
				String address = (String)iter.next();
				if (successfail.equals("0"))
				    retstring += " Disk operation for disk at " + address + " was successful\n";
				else
				    retstring += " Status of disk operation for disk at " + address + " cannot be determined";
			    }
			    return retstring;
			}
			//else
			  //  return super.result();
			return super.result();
		}
		
		/**
		 * Returns <tt>null</tt> or an exception
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	/**
	 * This adds a minidisk for a guest. Wraps
	 * {@link live.zvmtasks.ZVMTasks#createMDisk(String, String, String, String, String, String, String, String)}.
	 */
	static public class CreateMinidisk extends ThreadedUserAction
	{
		private final String strGuestName, strAllocationType, 
			strDiskSize, strDiskMode, strReadPW, strWritePW, strMultiPW;
		
		private Object virtualDeviceAddress;
		
		/**
		 * Creates an action that calls {@link live.zvmtasks.ZVMTasks#createMDisk(String, String, String, String, String, String, String, String)}.
		 *
		 * @param initInfo The ThreadInitInfo object
		 * @param strGuestName The name of the guest to create a minidisk for
		 * @param virtualDeviceAddress The address to create a minidisk at 
		 * @param strAllocationType How the minidisk is to be allocated (usually AUTOG)
		 * @param strDiskSizeMB The size of the disk in MB
		 * @param strDiskMode The access mode for the disk
		 * @param strReadPW The ReadPW for the disk (if RR mode)
		 * @param strWritePW The WritePW for the disk (if WR mode)
		 * @param strMultiPW The MultiPW for the disk (if MW mode and the other two passwords are specified)
		 */
		public CreateMinidisk(ThreadInitInfo initInfo,
				String strGuestName, Object virtualDeviceAddress,  
				String strAllocationType, String strDiskSizeMB, String strDiskMode, 
				String strReadPW, String strWritePW, String strMultiPW)
		{
			init(initInfo);
			
			this.strGuestName = strGuestName;
			this.virtualDeviceAddress = virtualDeviceAddress;
			this.strAllocationType = strAllocationType;
			this.strDiskSize = strDiskSizeMB;
			this.strDiskMode = strDiskMode;
			this.strReadPW = strReadPW;
			this.strWritePW = strWritePW;
			this.strMultiPW = strMultiPW;
		}
		
		/**
		 * Finds the lowest free virtual address for this guest
		 * @return String 4-digit hexadecimal address
		 */
		private String getVirtualAddress()
		{
			System.out.println("In getVirtualAddress()");
			if(virtualDeviceAddress instanceof String)
			{
				System.out.println("The address was specified directly");
				return (String) virtualDeviceAddress;
			}
			
			if(virtualDeviceAddress instanceof ThreadedZVMTasks.FindCommonFreeAddress)
			{
				String addy = 
					(String) ((ThreadedZVMTasks.FindCommonFreeAddress)virtualDeviceAddress).rawResult();
				System.out.println("The address was specified indirectly: " + addy);
				
				return addy; 
			}
			
			System.out.println("Address not specified. " + virtualDeviceAddress);
			return null;
		}
		
		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#createMDisk(String, String, String, String, String, String, String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.createMDisk(strGuestName, getVirtualAddress(), 
					strAllocationType, strDiskSize, strDiskMode, strReadPW, 
					strWritePW, strMultiPW);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Creating new minidisk for" + strGuestName + " at address " + 
				getVirtualAddress();
		}

		/**
		 * Returns <tt>null</tt> or an exception.
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This deletes a minidisk for a guest. Wraps
	 * {@link live.zvmtasks.ZVMTasks#deleteMDisk(String, String)}.
	 */
	static public class DeleteMinidisk extends ThreadedUserAction
	{
		private final String strGuestName, strVirtualDeviceAddress;
		/**
		 * Creates an action that will call {@link live.zvmtasks.ZVMTasks#deleteMDisk(String, String)}.
		 * @param initInfo The ThreadInitInfo object
		 * @param strGuestName The name of guest to delete a minidisk from
		 * @param strVirtualDeviceAddress Virtual address of minidisk to delete
		 */
		public DeleteMinidisk(ThreadInitInfo initInfo, String strGuestName, String strVirtualDeviceAddress)
		{
			init(initInfo);
			
			this.strGuestName = strGuestName;
			this.strVirtualDeviceAddress = strVirtualDeviceAddress;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#deleteMDisk(String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.deleteMDisk(strGuestName, strVirtualDeviceAddress);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Deleting a minidisk for" + strGuestName + " at address " + strVirtualDeviceAddress;
		}

		/**
		 * Returns <tt>null</tt> or an exception
		 * @return Object
		 */
		public Object rawResult()
		{
			return getException();
		}
	}
	
	/**
	 * This queries the minidisks for a guest. Wraps
	 * {@link live.zvmtasks.ZVMTasks#queryMDisks(String)}.
	 */
	static public class QueryMinidisks extends ThreadedUserAction
	{
		private final String strGuestName;
		LinkedList disks = null; 
		
		/**
		 * Creates an action that will call {@link live.zvmtasks.ZVMTasks#queryMDisks(String)}.
		 * @param initInfo The ThreadInitInfo object
		 * @param strGuestName The name of the guest whose minidisks to query
		 */
		public QueryMinidisks(ThreadInitInfo initInfo, String strGuestName)
		{
			init(initInfo);
			
			this.strGuestName = strGuestName;
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#queryMDisks(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			disks = tasks.queryMDisks(strGuestName);
		}
		
		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Querying Minidisks for " + strGuestName;
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.MDisk}
		 * or an exception or <tt>null</tt> if the call hasn't completed
		 * @return Object
		 */
		public Object rawResult()
		{
			if(disks != null)
				return disks;
			else
				return getException();
		}
	}
	
	/**
	 * This gets the saved segment groups that are on the server and their members.
	 * Wraps {@link live.zvmtasks.ZVMTasks#sharedStorageQuery()}. 
	 */
	static public class QuerySharedStorage extends ThreadedUserAction
	{
		private LinkedList segments = null;
	    
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#sharedStorageQuery()}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public QuerySharedStorage(ThreadInitInfo initInfo)
		{
			init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#sharedStorageQuery()}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			segments = tasks.sharedStorageQuery();
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Querying shared storage.";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.SavedSegment}s,
		 * an exception, or null if that call
		 * hasn't completed. 
		 * 
		 * @return A list of <tt>SavedSegment</tt> DTOs, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
			if(segments != null)
				return segments;
			else
				return getException();
		}
	}
	
	/**
	 * Grants a guest access to a shared storage segment
	 * Wraps (@link live.zvmtasks.ZVMTasks#sharedStorageAccessAdd(String, String)}.
	 */
	static public class SharedStorageAccessAdd extends ThreadedUserAction
	{
	    private String guestName;	
	    private String segmentName;
		
		/**
		 * Creates an action that will call
		 * (@link live.zvmtasks.ZVMTasks#sharedStorageAccessAdd(String, String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName Name of the guest to add
		 * @param segmentName Name of the segment to grant access for
		 */
		public SharedStorageAccessAdd(ThreadInitInfo initInfo, String guestName, String segmentName)
		{
		    this.guestName = guestName;
		    this.segmentName = segmentName;
		    init(initInfo);
		}

		/**
		 * Calls (@link live.zvmtasks.ZVMTasks#sharedStorageAccessAdd(String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.sharedStorageAccessAdd(guestName,  segmentName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Giving " + guestName + " access to shared segment " + segmentName + ".";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.SavedSegment}s,
		 * an exception, or null if that call
		 * hasn't completed. 
		 * 
		 * @return A list of <tt>SavedSegment</tt> DTOs, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
		    return getException();
		}
	}
	
	
	/**
	 * Revokes a guest's access to a shared storage segment
	 * Wraps (@link live.zvmtasks.ZVMTasks#sharedStorageAccessRemove(String, String)}.
	 */
	static public class SharedStorageAccessRemove extends ThreadedUserAction
	{
	    private String guestName;	
	    private String segmentName;
		
		/**
		 * Creates an action that will call
		 * (@link live.zvmtasks.ZVMTasks#sharedStorageAccessRemove(String, String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 */
		public SharedStorageAccessRemove(ThreadInitInfo initInfo, String guestName, String segmentName)
		{
		    this.guestName = guestName;
		    this.segmentName = segmentName;
		    init(initInfo);
		}

		/**
		 * Calls (@link live.zvmtasks.ZVMTasks#sharedStorageAccessRemove(String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.sharedStorageAccessRemove(guestName,  segmentName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Removing " + guestName + "'s access to " + segmentName;
		}

		/**
		 * Returns an exception, or null 
		 * 
		 * @return An exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
		    return getException();
		}
	}
	
	/** 
	 * Not Currently Used
	 * Queries whether a guest has access to that saved segment
	 * Wraps {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQuery(String, String)}.
	 */
	static public class SharedStorageAccessQuery extends ThreadedUserAction
	{
	    private String guestName;	
	    private String segmentName;
	
	    private Boolean acc = null;
	    
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQuery(String, String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param guestName The name of the guest to query
		 * @param segmentName The name of the segment to query
		 */
		public SharedStorageAccessQuery(ThreadInitInfo initInfo, String guestName, String segmentName)
		{
		    this.guestName = guestName;
		    this.segmentName = segmentName;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQuery(String, String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			acc = Boolean.valueOf( tasks.sharedStorageAccessQuery(guestName,  segmentName) );
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Setting access for shared storage.";
		}

		/**
		 * Returns a {@link java.lang.boolean} or an Exception or <tt>null</tt>
		 * 
		 * @return A boolean, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
		    if(acc != null) return acc;
		    else return getException();
		}
	}
	
	/**
	 * Gets the access name list for a particular saved segment
	 * Wraps {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQueryNameList(String)}.
	 */
	static public class SharedStorageAccessQueryNameList extends ThreadedUserAction
	{
	    private String segmentName;
	
	    private LinkedList guests = null;
	    
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQueryNameList(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param segmentName Name of the segment to query
		 */
		public SharedStorageAccessQueryNameList(ThreadInitInfo initInfo, String segmentName)
		{
		    this.segmentName = segmentName;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#sharedStorageAccessQueryNameList(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			guests = tasks.sharedStorageAccessQueryNameList(segmentName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Getting access info for shared storage segment " + segmentName + ".";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.SavedSegment}s,
		 * an exception, or null if that call
		 * hasn't completed. 
		 * 
		 * @return A list of <tt>SavedSegment</tt> DTOs, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
		    if(guests!= null) return guests;
		    else return getException();
		}
	}
	
	/**
	 * Creates a saved segment
	 * Wraps {@link live.zvmtasks.ZVMTasks#sharedStorageCreate(String, String, LinkedList)}.
	 */
	static public class SharedStorageCreate extends ThreadedUserAction
	{
	    private String segmentName;
	    private String segSizeInMB;
	    private LinkedList guestsToAdd;	
	
	    private Boolean acc = null;
	    
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#sharedStorageCreate(String, String, LinkedList)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param segmentName The name of the new saved segment 
		 * @param segSizeInMB The size of the new saved segment
		 * @param guestsToAdd Guests to authorize for the segment
		 */
		public SharedStorageCreate(ThreadInitInfo initInfo, String segmentName, String segSizeInMB, LinkedList guestsToAdd)
		{
		    this.segmentName = segmentName;
		    this.segSizeInMB = segSizeInMB;
		    this.guestsToAdd = guestsToAdd;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#sharedStorageCreate(String, String, LinkedList)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.sharedStorageCreate(segmentName, segSizeInMB, guestsToAdd);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Creating shared storage " + segmentName + ".";
		}

		/**
		 * Returns a {@link java.util.LinkedList} of {@link live.dto.SavedSegment}s,
		 * an exception, or null if that call
		 * hasn't completed. 
		 * 
		 * @return A list of <tt>SavedSegment</tt> DTOs, an exception, or <tt>null</tt>
		 */
		public Object rawResult()
		{
		    if(acc != null) return acc;
		    else return getException();
		}
	}
	
	/**
	 * Deletes a saved segment
	 * Wraps {@link live.zvmtasks.ZVMTasks#sharedStorageDelete(String)}.
	 */
	static public class SharedStorageDelete extends ThreadedUserAction
	{

	    private String segmentName;
	
		/**
		 * Creates an action that will call
		 * {@link live.zvmtasks.ZVMTasks#sharedStorageDelete(String)}.
		 * 
		 * @param initInfo The ThreadInitInfo object
		 * @param segmentName Name of the segment to delete
		 */
		public SharedStorageDelete(ThreadInitInfo initInfo, String segmentName)
		{
		    this.segmentName = segmentName;
		    init(initInfo);
		}

		/**
		 * Calls {@link live.zvmtasks.ZVMTasks#sharedStorageDelete(String)}.
		 */
		protected void runImpl() throws ZVMTasksException
		{
			tasks.sharedStorageDelete(segmentName);
		}

		/**
		 * Returns a description of this task
		 * @return String
		 */
		public String description()
		{
			return "Deleting shared storage " + segmentName + ".";
		}

		/**
		 * Returns an exception, or null. 
		 * 
		 * @return Object
		 */
		public Object rawResult()
		{
		    return getException();
		}
	}

	
} // class ThreadedZVMTasks
