package live.threads;

import java.util.Date;
import live.zvmtasks.ZVMTasks;
import live.cache.Cache;
import live.dto.ConnectionInfo;

/**
 * This function serves as a base class for all of the classes that will
 * actually do useful work. For each action you want the user to be able to do,
 * subclass this and implement three functions:
 * <ol>
 * <li>{@link #runImpl()}should perform the action</li>
 * <li>{@link #description()}should return a one-line summary of what is being
 * done</li>
 * <li>{@link #result()}should return the, well, result of the operation once
 * it is finished, or null if it is sitll ongoing.</li>
 * </ol>
 * When you want to start the task in a new thread, call one of
 * {@link #startJoinForever()},{@link #startJoinTimeout(int)}, or
 * {@link #startNoJoin()}. (You could also call the inherited
 * {@link Thread#start()}or {@link #run()}, but in any application that uses
 * any of the three custom functions it is recommended that the programmer use
 * only those three to be consistant and even with the naming conventions.)
 * 
 * Because most of what the subclass will be doing will require parameters, you
 * have to set them in fields before calling {@link Thread#start()}, just like
 * you do with any other thread.
 * 
 * Finally, for any subclasses that use the ZVMTasks library, call
 * {@link #init(ThreadInitInfo)}before starting the action. (Probably do this
 * in the constructor.) Then your subclass can simply use the protected field
 * {@link #tasks}to call the method(s). You could go through the process of
 * making your subclass keep track of the appropriate info, but why do that when
 * it's done for you? It is okay to pass {@link #init(ThreadInitInfo)}a null
 * parameter, or for either of the pieces of information in
 * {@link ThreadInitInfo}to be null, but it's expected that this will be the
 * case only in testing (for instance, the {@link WaitAction}class) and
 * possibly other exceptional situations.
 */
public abstract class ThreadedUserAction extends Thread
{
	/**
	 * Sets the object that tracks the threads.
	 * 
	 * @param initInfo The object with server login information and thread lists
	 * @param list Should this thread be included in the ActiveThreadsInfo lists
	 */
	public final void init(ThreadInitInfo initInfo, boolean list)
	{
		thisThreadsID = ThreadedUserAction.nextFreeID.nextID();
		
		done = false;
		track = list;

		if(initInfo != null)
		{
			activeThreadLists = initInfo.getActiveThreadsInfo();
			ConnectionInfo connection = initInfo.getConnectionInfo();
			cache = initInfo.getCache();

			if(connection != null)
			{
				tasks = new ZVMTasks(connection.getHostName(), connection
						.getPort(), connection.getUserName(), connection
						.getPassword());
			}
		}
	}

	/**
	 * Sets the object that tracks the threads.
	 * 
	 * @param initInfo The object with server login information and thread lists
	 */
	public final void init(ThreadInitInfo initInfo)
	{
		init(initInfo, true);
	}

	/**
	 * This method returns the time at which the thread was started. (Since it
	 * would technically be possible for a thread to be started multiple times
	 * from the same object, it returns the last time the thread was started.)
	 * 
	 * @see #getTimeFinished()
	 * @return The time at which the thread was last started
	 */
	public Date getTimeStarted()
	{
		return (Date) timeStarted.clone();
	}

	/**
	 * This method returns the time at which the thread finished its task and
	 * died. (Like {@link #getTimeStarted()}, technically this is the last time
	 * the thread died.
	 * 
	 * @see #getTimeStarted()
	 * @return The time at which the thread last died
	 */
	public Date getTimeFinished()
	{
		return (Date) timeEnded.clone();
	}

	/**
	 * This method returns a friendly description of the action the object
	 * represents. This is more meant for errors and such (not SMAPI errors,
	 * coding errors) where the {@link #toString()}method is called
	 * automatically, so the description is prepended with the class name. For
	 * the description without this prefix, such as for displaying the status
	 * page, use the abstract {@link #description()}method instead.
	 * 
	 * @see #description()
	 * @return A human-readable summary of the action being performed, prefixed
	 *         with "ThreadedUserAction: "
	 */
	public String toString()
	{
		return "ThreadedUserAction: " + description();
	}

	/**
	 * This should be not called by hand under most circumstances [OPTIONAL:
	 * except for queries]. This function is mostly a possthrough to the
	 * abstract {@link #runImpl()}, but also sets the starting and ending
	 * times. Instead of calling this, you probably want to call
	 * {@link Thread#start()}, which will start a new thread and begin
	 * executing {@link #run()}there.
	 * 
	 * @see #runImpl()
	 * @see Thread#start()
	 */
	public void run()
	{
		done = false;
		if(activeThreadLists != null)
			activeThreadLists.markInProgress(this);
		// The default Date object holds the time now
		timeStarted = new Date();
		
		
		setInProgress();
		
		try
		{
			runImpl();
			
			if(this.getStatus() != ERROR && getStatus() != CANCELED)
				setComplete();
		}
		catch(Exception e)
		{
			exception = e;
			setError(e);
		}
		
		timeEnded = new Date();
		if(activeThreadLists != null)
			activeThreadLists.markComplete(this);
		done = true;
	}

	/**
	 * This function should be implemented to do whatever task the class was
	 * created to do. You shouldn't call this function yourself; it's called
	 * automatically by {@link #run()}. (But you probably shouldn't call
	 * {@link #run()}yourself either.)
	 * 
	 * @see #run()
	 */
	protected abstract void runImpl() throws Exception;

	/**
	 * This method returns a friendly description of the action the object
	 * represents. Unlike the {@link #toString()}method, this is meant to be
	 * used in, for example, the status page and anywhere else where the end
	 * user will see it.
	 * 
	 * @see #toString()
	 * @return A human-readable summary of the action being performed
	 */
	public abstract String description();

	/**
	 * This returns the result in some manner or other. It should be suitable
	 * for display in the status page. May return HTML that can appear as the
	 * child of TD tags.
	 * @return String result
	 */
	public String result()
	{
		return resultString;
	}

	/**
	 * This returns the result of the call in a manner that is defined by the
	 * subclass. It should probably simply pass through the return value from
	 * the ZVMTasks call.
	 * 
	 * @return The result of the action. (See subclasses for individual
	 *         definitions.)
	 */
	public abstract Object rawResult();

	/**
	 * This function begins the action, returning when it completes. It is
	 * equivalent to calling either: 1.{@link #run}(which does not start a new
	 * thread, avoiding the oveheard of going to the JVM; the current
	 * implementation), or 2.{@link Thread#start()}then {@link Thread#join()},
	 * which starts the thread and then joins the calling and action threads
	 * until the latter dies.
	 * 
	 * (Actually, the two methods are not exactly equivalent, as you can
	 * interrput a thread that's running, causing join to throw an
	 * InterruptedException exception. There is no equivalent to this with #1
	 * above. However, I'm pretty sure this exception should never be raised.)
	 */
	public void startJoinForever()
	{
		run();

		//start();
		//join();
	}

	/**
	 * This function begins the action, returning when it completes, unless it
	 * takes more than timeout_ms milliseconds, in which case it returns at that
	 * point.
	 * 
	 * @return True if the thread finished before the timeout expired
	 */
	public boolean startJoinTimeout(int timeout_ms)
	{
		start();
		
		try
		{
			join(timeout_ms);
		}
		catch(InterruptedException e)
		{}
		
		return isFinished();
	}

	/**
	 * Whether the thread is finished.
	 * @return boolean
	 */
	public boolean isFinished()
	{
		return !(getTimeFinished() == null);
	}

	/**
	 * This function begins the action, returning immediately. It is equivalent
	 * to simply calling {@link Thread#start()}.
	 */
	public void startNoJoin()
	{
		start();
	}

	private Date timeStarted;

	private Date timeEnded;

	private ActiveThreadsInfo activeThreadLists;
	private Cache cache;

	/**
	 * If {@link #init(ThreadInitInfo)}has been called with a non-null argument
	 * for which {@link ThreadInitInfo#getConnectionInfo()}is non-null, this
	 * will reference a ZVMTasks object which will make the connection to the
	 * server with the authentication parameters defined in
	 * {@link #init(ThreadInitInfo)}'s connectionInfo#getConnectionInfo().
	 */
	protected ZVMTasks tasks;

	protected boolean done;

	private boolean track;

	/**
	 * Returns the ID of the thread
	 * @return int Thread's ID
	 */
	public int getID()
	{
		return thisThreadsID;
	}
	
	/** This is a wrapper class for an int; {@link java.lang.Integer} won't
	 * work because you can't change it; See the init function for
	 * a discussion of why this is (or rather, may be) necessary.
	 */
	private static final class IDFactory
	{
		private int value;
		
		/**
		 * Initializes the value to 0.
		 */
		public IDFactory()
		{
			value = 0;
		}
		
		/**
		 * Returns and then increments value.
		 * @return int Current ID value
		 */
		synchronized public int nextID()
		{
			return value++;
		}
	}
	
	static private IDFactory nextFreeID = new IDFactory();
	private int thisThreadsID = -1;
	private Exception exception = null;
	
	public static final int PENDING     = 0;
	public static final int IN_PROGRESS = 1;
	public static final int COMPLETE    = 2;
	public static final int ERROR       = 3;
	public static final int CANCELED    = 4;
	public static final int INDETERMINATE = 5;
	
	private int status = 0;
	private String resultString = "Pending";
	
	/**
	 * Sets status and resultString to In Progress.
	 */
	public void setInProgress()
	{
		status = IN_PROGRESS;
		resultString = "In progress...";
	}
	
	/**
	 * Sets status and resultString to Complete.
	 */
	public void setComplete()
	{
		status = COMPLETE;
		resultString = "Complete: Success";
	}
	
	/**
	 * Sets status and resultString to Indeterminate.
	 */
	public void setIndeterminate()
	{
		setIndeterminate(null);
	}
	
	/**
	 * Sets the status of a task to indeterminate and the resultString to a specified message
	 * @param message Message to display
	 */
	public void setIndeterminate(String message)
	{
		status = INDETERMINATE;
		resultString = "Task status is unknown";
		if(message != null)
			resultString += " (" + message + ")";
	}
	
	/**
	 * Sets the status and resultString to Error
	 */
	public void setError()
	{
		status = ERROR;
		resultString = "Error, unspecifed";
	}
	
	/**
	 * Sets the status to error and the resultString to a specified message
	 * @param e Exception to get the error message from
	 */
	public void setError(Exception e)
	{
		e.printStackTrace();
		status = ERROR;
		if(e.getMessage() != null)
			resultString = "Error: " + e.getMessage();
		else
			resultString = "Error. Null message, exception class is " + e.getClass().getName();
	}
	
	/**
	 * Sets the status and resultString to canceled.
	 */
	public void setCanceled()
	{
		status = CANCELED;
		resultString = "Canceled";
	}
	
	/**
	 * Returns the status.
	 * @return int status
	 */
	public int getStatus()
	{
		return status;
	}
	
	/**
	 * Returns the cache
	 * @return Cache 
	 */
	protected Cache getCache()
	{
		while(cache.getState() == Cache.UPDATING)
		{
			// Poll the cache until it's not updating
			try   { sleep(1000); }
			catch (InterruptedException e) {}
		}
		
		return cache;
	}
	
	/**
	 * Returns the exception thrown
	 * @return Exception
	 */
	public Exception getException()
	{
		return exception;
	}
}
