/*
 * Created on Jul 15, 2005
 */
package live.threads;

import java.util.Iterator;
import java.util.LinkedList;

/**
 * This class is a container that holds ThreadedUserActions that should be run
 * later. It is itself a ThreadedUserAction, so has run(), start(), etc. methods.
 * In this case, when they are called, the runner should kick off all the tasks
 * that it has stored. There are (TODO) two concrete subclasses of this,
 * SequentialRunner and ParallelRunner, based upon whether the tasks can be run
 * in parallel or not. (Note that the distinction is NOT upon whether they 
 * actually <em>do</em> run in parallel, just whether they can. See the docs for
 * each of these for more detail, especially ParallelRunner.)
 * 
 * To use this, make an object of one of the concrete runners, call
 * {@link #schedule(ThreadedUserAction)} for each task you want to run,
 * then call {@link ThreadedUserAction#start()} or any of the other
 * functions that will start the thread.
 * 
 * To retrieve the results, call {@link #rawResult()}. This returns a linked
 * list of the <tt>ThreadedUserActions</hh> that were passed to 
 * <tt>schedule(...)</tt> before. You can iterate through this list and call 
 * {@link ThreadedUserAction#rawResult()} on each of them. Also,
 * {@link #result()} should return an error message if an exception is thrown
 * by one of the threads (subject to its implementation by a subclass).
 * 
 * 
 * @author Evan Driscoll
 */
public abstract class Runner extends ThreadedUserAction
{
	/**
	 * This stores occurances of exceptions and the task that 
	 * caused them so they can be reported
	 */
	/*static public class ExceptionRecord
	{
		Exception e;
		ThreadedUserAction task;
	}*/

	//private LinkedList exceptions = new LinkedList();
	private String description;
	
	/**
	 * This holds a modified version of the ThreadInitInfo object that
	 * was passed in. In order that every subaction not show up in the
	 * status page, the session's threadinfo object can't be used. So
	 * {@link #init(ThreadInitInfo, String)} takes the ThreadInitInfo 
	 * passed in, and makes a new one local to the runner with the same 
	 * ConnectionInfo and a different ActiveThreadsInfo. (TODO: make teh local Active object)
	 */
	private ThreadInitInfo modifiedInfo;
	
	protected LinkedList taskList = new LinkedList();
	protected ThreadInitInfo initInfo;
	
	/**
	 * Does the trickery necessary in order to create a new ThreadInitInfo
	 * object local to the thread. Also, stores the description. This is
	 * not really intended to be called publically, just by derived classes.
	 * 
	 * @param info
	 * @param description
	 */
	public final void init(ThreadInitInfo info, String description)
	{
		init(info, description, true);
	}
	
	/**
	 * Does the trickery necessary in order to create a new ThreadInitInfo
	 * object local to the thread. Also, stores the description. This is
	 * not really intended to be called publicly, just by derived classes.
	 * 
	 * @param info
	 * @param description
	 * @param removeLists
	 */
	public final void init(ThreadInitInfo info, String description, boolean removeLists)
	{
		ActiveThreadsInfo i = removeLists ? null : info.getActiveThreadsInfo(); 
		modifiedInfo = new ThreadInitInfo(info.getConnectionInfo(), 
				i, info.getCache()); 
		
		super.init(info);
		initInfo = info;
		
		this.description = description;
	}
	
	/**
	 * This adds <tt>action</tt> to the list of subactions that will be run 
	 * 
	 * @param action The action to schedule
	 */
	public void schedule(ThreadedUserAction action)
	{
		action.init(modifiedInfo);
		taskList.add(action);
	}

	/**
	 * This returns the description set in init
	 * 
	 * @see live.threads.ThreadedUserAction#description()
	 */
	public String description()
	{
		return description;
	}

	/**
	 * Returns the tasks that are to be/are being/have been done
	 */
	public Object rawResult()
	{
		return taskList;
	}
	
	/**
	 * Sets the thread as canceled
	 */
	public void setCanceled()
	{
		super.setCanceled();
		
		Iterator iter = taskList.iterator();
		while(iter.hasNext())
		{
			ThreadedUserAction t = (ThreadedUserAction) iter.next();
			if(t.getStatus() == ThreadedUserAction.PENDING)
				t.setCanceled();
		}
	}
}
