package live.threads;

import java.util.LinkedList;

/**
 * This class tracks what threads are currently active. When a thread begins
 * processing (begins its run), it should call
 * {@link #markInProgress(ThreadedUserAction)}. When a thread ends its
 * processing, it should call {@link #markComplete(ThreadedUserAction)}. Both
 * functions take one parameter, which is the thread that is starting or
 * finishing. (So essentially <code>this</code> from whichever thread is
 * running.)
 * 
 * In addition, you can get a list of the threads that are currently running,
 * and a list of those that have been completed since the last call. TODO: A
 * longer list into the past
 * 
 * <b>When synchronizing on multiple objects, it is imperative that one do it in
 * the order inProgress, then recentlyComplete, then completedJobs in order to
 * avoid deadlocks. If only synchronizing on two of the above, you still must do
 * it in the same order. </b> Client code need not lock on recentlyComplete at
 * all, but if using both the inProgress or completeJobs objects synchronization
 * is a must (and in that order if using both).
 *  
 */
public class ActiveThreadsInfo
{
	/**
	 * Marks the thread as currently running
	 * 
	 * @param t The {@link ThreadedUserAction}that is starting to run
	 */
	public void markInProgress(ThreadedUserAction t)
	{
		synchronized(inProgress)
		{
			inProgress.add(t);
		}
	}

	/**
	 * Marks the thread as ending. In other words, it moves it from the list of
	 * in-progress threads to done threads. If the thread took longer than
	 * {@link #timeThreshold} milliseconds to run, adds it to the recently
	 * complete list too. 
	 * 
	 * @param t The {@link ThreadedUserAction}that is starting to run
	 */
	public void markComplete(ThreadedUserAction t)
	{
		synchronized(inProgress)
		{
			synchronized(recentlyComplete)
			{
				synchronized(completedJobs)
				{
					inProgress.remove(t);

					if((t.getTimeFinished().getTime() - t.getTimeStarted()
							.getTime()) >= timeThreshold)
					{
						// This sorts them so that threads with errors will
						// appear before threads without errors. They will be
						// ordered in opposite manners with respect to time
						// finished, but that's not a big deal IMO.
						
						if(t.getStatus() == ThreadedUserAction.ERROR)
							recentlyComplete.addFirst(t);
						else
							recentlyComplete.addLast(t);
					}

					completedJobs.add(t);
				}
			}
		}
	}

	/**
	 * This function returns a list of threads that have ended (called
	 * markComplete) since the last time it was called.
	 * 
	 * There is no synchronization issue outside of this function; calling code
	 * has free reign over the use of it, and the list and all the threads in it
	 * will be GC'd after it loses its reference to it. (Currently. Do not
	 * depend on this GCing occuring.)
	 * 
	 * @return A list of {@link ThreadedUserAction}s passed to
	 *         {@link #markComplete}since the last call
	 */
	public LinkedList getRecentlyComplete()
	{
		synchronized(recentlyComplete)
		{
			LinkedList ret = recentlyComplete;
			recentlyComplete = new LinkedList();
			return ret;
		}
	}

	/**
	 * This function returns a list of threads that have been completed sometime
	 * somewhat sort of recently. This is deliberately vague.
	 * 
	 * @return A list of {@link ThreadedUserAction}s that have been completed
	 *         in the past
	 */
	public LinkedList getCompletedJobs()
	{
		return completedJobs;
	}

	/**
	 * This function returns a list of threads that are currently running (have
	 * not called markComplete).
	 * 
	 * Here (as opposed to {@link #getRecentlyComplete}there IS a
	 * synchronization issue, and calling code must synchronize on it. This
	 * function does not want to take the time to duplicate the list to return
	 * while potentially blocking other threads from executing, but it needs
	 * continued use of the list.
	 * 
	 * @return A list of {@link ThreadedUserAction}s passed to
	 *         {@link #markInProgress}but not {@link #markComplete}
	 */
	public LinkedList getInProgress()
	{
		return inProgress;
	}

	private LinkedList inProgress = new LinkedList();
	private LinkedList recentlyComplete = new LinkedList();
	private LinkedList completedJobs = new LinkedList();

	/**
	 * The length of time a thread must be active for before it will be added to
	 * the recently complete list. This is so that every little query and whatnot
	 * will not show up in the header.
	 */
	private int timeThreshold = live.struts.actions.Util.TIMEOUT_MS;
	
	/**
	 * This sets {@link #timeThreshold}.
	 * 
	 * @param newThreshold The new threshold value
	 */
	public void setTimeThreshold(int newThreshold)
	{
		timeThreshold = newThreshold;
	}
}
