/*
 * Created on Jul 8, 2005
 */
package live.struts.actions;

import org.apache.struts.Globals;
import org.apache.struts.action.ActionErrors;
import org.apache.struts.action.ActionForward;

/**
 * This is another sort of "convienience base class" that replaces a fairly
 * common process. There are a number of actions the user can take that require
 * both a page to be displayed to get some information then to actually carry
 * out the action. For instance, when creating a network, an action is needed to
 * get the list of guests on the server so the user can select what guests to
 * add, then another action is needed to do the actual addition. As another
 * example, deleting a guest displays a confirmation page before deleting.
 * 
 * In order to reduce the proliferation of Actions, things such as these are
 * combined into one Action class which contains a method for each. The
 * sever decides which function to run based on whether the request was GETed
 * or POSTed. In the former case, the Action should simply gather any information
 * necessary and display the page to the user; in the latter, it should actually
 * carry out the request.
 * 
 * This class takes care of the determination of which method was used in the
 * request and calling the appropriate method.
 * 
 * So an example with delete guest works as follows:
 * <ol>
 * <li>The user, from the individual guest page, clicks delete</li>
 * <li>The {@link DeleteGuestAction} action (a subclass of this) is called, 
 * {@link #act(CheckedAction.ActionInfo)} determines that it was submitted using GET and
 * calls {@link #displayPage(CheckedAction.ActionInfo)}.</li>
 * <li>{@link DeleteGuestAction#displayPage(CheckedAction.ActionInfo)} sets
 * some attributes indicating what the user is doing and forwards to the
 * generic confirmation page, which is displayed to the user.</li>
 * <li>When the user clicks on the "yes, I want to delete who-and-who" link,
 * the confirmation page POSTs to the same URL it came from. This calls this
 * Action again, but this time <tt>act</tt> figures out that it was a POST
 * request, so calls {@link #performAction(CheckedAction.ActionInfo)}.</li>
 * <li>{@link DeleteGuestAction#displayPage(CheckedAction.ActionInfo)} then
 * actually deletes the guest.</li>
 * 
 * @author Evan Driscoll
 */
abstract public class SwitchedAction extends CheckedAction
{

	/**
	 * <tt>act</tt> is final so that subclasses don't override it by mistake
	 * and break things.
	 * 
	 * @see live.struts.actions.CheckedAction#act(live.struts.actions.CheckedAction.ActionInfo)
	 */
	public final ActionForward act(ActionInfo info)
	{
		/* There are three paths for a request. If a GET request comes in,
		 * it goes directly to #displayPage(ActionInfo) via the overall else
		 * below.
		 * 
		 * If a POST request comes in, we validate it. (Why we don't let Struts
		 * I'll get to in a sec.) If the form validates, we perform the action
		 * via the if branch of both ifs. If the form doesn't validate, we
		 * call displayPage.
		 * 
		 * The reason this is necessary is because of the way we use the same
		 * URL for both displaying the page and performing the action. In the
		 * struts-config file, actions that use struts validation need an input
		 * attribute that tell struts what page to forward to if the validation
		 * fails. There are sort of two options for what to put in this, neither
		 * of which work.
		 *   1. If you put the JSP in there, it fails for any pages that use 
		 *      dynamic information. The submission is a new request, so any
		 *      attributes that were set before don't exist any more.
		 * 
		 *   2. If you put the action URL in there, the sequence of events
		 *      is roughly as follows:
		 *         1. A bad submission comes in.
		 *         2. Struts sees that the input needs to be validated
		 *         3. Struts tries to validate it, but it fails
		 *         4. Struts forwards the request to the input URL
		 *         5. Struts sees that the input needs to be validated
		 *         6. Struts tries to validate it, but it fails
		 *         7. Struts forwards the request to the input URL
		 *         8. Struts sees that the input needs to be validated
		 *            [ steps 9 through 765 omitted ]
		 *       766. You get an eight-page stack trace
		 */
		if(info.request.getMethod().equalsIgnoreCase("POST"))
		{
			ActionErrors errors = 
				(info.form != null ? 
				 info.form.validate(info.mapping, info.request) : 
				 null);
			
			if(errors == null || errors.isEmpty())
				return performAction(info);
			else
			{
				info.setRequestAttribute(Globals.ERROR_KEY, errors);
				return displayPage(info);
			}
		}
		else
			return displayPage(info);
	}

	/**
	 * This function is called when a request was submitted by an HTTP GET
	 * request. Implement this in subclasses to display a page where the
	 * user can enter any information necessary to actually do something,
	 * confirm that they want to actually do something, etc.
	 * 
	 * @param info The request info passed to {@link #act(CheckedAction.ActionInfo)}
	 * @return The forward to go to
	 */
	abstract public ActionForward displayPage(ActionInfo info);

	/**
	 * This function is called when a request was submitted by an HTTP POST
	 * request. This function should actually perform whatever action the
	 * user wanted. 
	 * 
	 * You should probably strive to make it so that the URL that
	 * calls this action can only be POSTed to from the page displayed by
	 * {@link #displayPage(CheckedAction.ActionInfo)} so that you know that the user is
	 * coming from that page.
	 * 
	 * @param info The request info passed to {@link #act(CheckedAction.ActionInfo)}
	 * @return The forward to go to
	 */
	abstract public ActionForward performAction(ActionInfo info);
}
