/*
 * Created on Jul 25, 2005
 */
package live.struts.actions;

import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

import org.apache.struts.action.ActionForward;

import live.constants.Attributes;
import live.constants.Mappings;
import live.threads.ParallelRunner;
import live.threads.SequentialRunner;
import live.threads.ThreadedZVMTasks;
import live.struts.forms.CloneForm;


/**
 * @author Evan Driscoll
 */
public class CloneGuestAction extends SwitchedAction
{

	/**
	 * @see live.struts.actions.SwitchedAction#displayPage(live.struts.actions.CheckedAction.ActionInfo)
	 */
	public ActionForward displayPage(ActionInfo info)
	{
		info.setRequestAttribute("guestName", info.getRequestParameter("guestName"));
		return info.findForward(Mappings.guests.clone.showPage);
	}

	/**
	 * @see live.struts.actions.SwitchedAction#performAction(live.struts.actions.CheckedAction.ActionInfo)
	 */
	public ActionForward performAction(ActionInfo info)
	{
		CloneForm form = (CloneForm) info.form;
		
		// Common
		String existGuestName = info.getRequestParameter("guestName");
		String existGroupName = info.cache.getGuest(existGuestName).getGroupName();
		
		String password = form.getPassword();
		boolean copyDisks = form.cloneDisks();
		boolean formatDisks = form.format();
		
		boolean multiple = form.makeMultiple();
		
		Collection names = null;
		
		if(multiple)
		{
			try
			{
				String basenamefield = form.getCloneName();
				basenamefield = basenamefield.toUpperCase();
				
				int numclones = form.numberClones();
				int startnum = form.startIndex();
				
				names = generateMultipleGuestNamesList(basenamefield, startnum, startnum+numclones );
			}
			catch(Exception e)
			{
				info.setRequestAttribute(Attributes.error.message, e.getMessage());
				return info.findForward(Mappings.guests.clone.error); 
			}
		}
		else
		{
			String namefield = form.getCloneName();
			namefield = namefield.toUpperCase();
			
			names = new LinkedList();
			names.add(namefield);
		}
		
		System.out.println("Cloning " + existGuestName);
		
		ParallelRunner mainThread = new ParallelRunner(info.threadInfo, 
				"Cloning guest " + existGuestName, 1);
		
		Iterator iter = names.iterator();
		while(iter.hasNext())
		{
			String name = (String) iter.next();
			
			System.out.println("  as guest " + name);
			
			SequentialRunner guest = new SequentialRunner(info.threadInfo, "Cloning");
			
			ThreadedZVMTasks.CloneGuest clone = 
				new ThreadedZVMTasks.CloneGuest(
					info.threadInfo, existGuestName, name, password, copyDisks, formatDisks);
			guest.schedule(clone);
			
			ThreadedZVMTasks.SetGuestToGroup group =
				new ThreadedZVMTasks.SetGuestToGroup(info.threadInfo,
						name, null, existGroupName);
			guest.schedule(group);
			
			mainThread.schedule(guest);
		}
		
		mainThread.startJoinTimeout(Util.TIMEOUT_MS);
		
		return info.smartForward(mainThread, Mappings.guests.list.This);
	}


    /**
     * This is the function we use to generate the names for a family of clones.
     * 
     * @param startText The base guest name
     * @param startNumber The number at which the clone family will start
     * @param endNumber The number at which the clone family will end
     * @return LinkedList LinkedList of names for clones
     * @throws Exception Exception if there is an error with the parameters
     */
    private LinkedList generateMultipleGuestNamesList(String startText, int startNumber, int endNumber)
	throws Exception
    {
	//Sanity check args
	if (startNumber > endNumber) throw new Exception("generateMultipleGuestNamesList: developer error! endNumber must be greater than startNumber");
	if (startNumber < 0) throw new Exception("generateMultipleGuestNamesList: developer error! Startnumber must be > 0");
	if (endNumber < 0) throw new Exception("generateMultipleGuestNamesList: developer error! Startnumber must be > 0");
	
	//Ensure we are creating names 8 chars or less.
	int charCount = startText.length() + countDigits(endNumber);
	if (charCount > 8) throw new  Exception("generateMultipleGuestNamesList: You are trying to create guests with names over eight characters long.  Please reduce the length of you guests's names."); 
	LinkedList guestList = new LinkedList();
	
	//Length we need.
	int lengthNeeded = countDigits(endNumber);
	
	int curNumber = startNumber;
	while(curNumber <= endNumber)
	{
	    guestList.addLast(startText + formatFixedNumberString(curNumber, lengthNeeded));
	    curNumber++;
	}
	
	return guestList;
    }
    
    /**
     * Puts the number string in the proper format for the clone name 
     *   (ensures that all clone names will have the same number of characters).
     * @param number The number to format.
     * @param totalDesiredDigitCount How many digits to make the number string.
     * @return String The now-formatted number.
     * @throws Exception if the number is not able to be formatted properly.
     */
    private String formatFixedNumberString(int number, int totalDesiredDigitCount)
	throws Exception
    {
	//Make sure args have acceptable values
	if(totalDesiredDigitCount < 1 || totalDesiredDigitCount > 8) throw new Exception("formatFixedNumberString: Developer error! totalDesiredDigitCount should have value between 1 - 8");
	if(number < 0) throw new Exception("formatFixedNumberString: Developer error! number should be non-negative");
	
	//Count digits
	int curDigitCount = countDigits(number);
	
	//Parse to String
	String numberString = Integer.toString(number);
	
	//Sanity check
	if (curDigitCount > totalDesiredDigitCount) throw new Exception("formatFixedNumberString: Developer error! curDigitCount greater than totalDesiredDigitCount");
	
	while(totalDesiredDigitCount > curDigitCount)
	{
	    numberString = "0" + numberString;
	    curDigitCount++;
	}
	
	return numberString;
    }
    
    /**
     * Given a number, this function will count the number of digits that are
     *   present in the number; number=100 would return 3, number=2 would return 1.
     * @param number The number for which we wish to know the number of digits.
     * @return The number of digits in number.
     * @throws Exception If the number is negative.
     */
    private int countDigits(int number)
	throws Exception
    {
	//Sanity check, we only work on positive numbers.
	if(number < 0) throw new Exception("countDigits: Developer error! number should be non-negative");
	
	//0 is a special case, so we'll handle by hard coding for 1-digit case.
	if(number < 10) return 1;
	
	int count=0;
	while( number > 0)
	{
	    number /= 10; //Strip off lowest digit 
	    count++;
	}
	
	return count;
    }

}
