/*
 * Created on Jul 8, 2005
 */
package live.struts.actions;

import java.net.URLEncoder;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;

import live.cache.Cache;
import live.constants.Attributes;
import live.constants.Mappings;
import live.threads.ThreadInitInfo;
import live.threads.ThreadedUserAction;

/**
 * This class takes care of much of the tedious mucking about that we used to
 * have to do at the beginning of every Action. Specifically, it:
 * <ul>
 * <li>Sets the request attribute listing recently done threads (for the
 * header) if there are any,</li>
 * <li>checkes to see if the session has expired and, if so, forwards the user
 * to the login page,</li>
 * <li>retreives the thread info from the session, casting it to the
 * appropriate type,</li>
 * <li>retreives the cache from the servletContext, casting it to a cache
 * object,</li>
 * <li>and checkes to see if the cache is valid and, if not, forwards them to
 * an error.</li>
 * </ul>
 * 
 * You use this class the same way you would
 * {@link org.apache.struts.action.Action}, except you implement the
 * {@link #act(CheckedAction.ActionInfo)} method instead of <tt>execute</tt>.
 * 
 * Rather than pass all four parameters we had before plus the cache plus
 * whatever else we may think is useful in the future to the subclasses, there
 * is a class used like a C struct ({@link ActionInfo}) that we put all that
 * in and just pass that.
 * 
 * @author Evan Driscoll
 */
abstract public class CheckedAction extends org.apache.struts.action.Action
{
	public final static class ActionInfo
	{
		// These four are exactly the parameters normally passed to execute
		public ActionMapping mapping;
		public ActionForm form;
		public HttpServletRequest request;
		public HttpServletResponse response;

		/** The thread info from the session */
		public ThreadInitInfo threadInfo;

		/** The cache from the servlet context */
		public Cache cache;

		/** Makes an ActionInfo with the specified field names */
		public ActionInfo(ActionMapping mapping, ActionForm form,
				HttpServletRequest request, HttpServletResponse response,
				ThreadInitInfo tii, Cache cache)
		{
			this.mapping = mapping;
			this.form = form;
			this.request = request;
			this.response = response;
			this.cache = cache;
			this.threadInfo = tii;
		}

		/**
		 * This is a passthrough to the request's
		 * {@link javax.servlet.http.HttpServletRequest#getParameter(String)}
		 * function. It's just for source prettyness than anything, so rather
		 * than say myActionInfo.request.getParameter you can say
		 * myActionInfo.getRequestParameter
		 * 
		 * @param param The request parameter to get
		 * @return The value of that request parameter.
		 */
		String getRequestParameter(String param)
		{
			return request.getParameter(param);
		}

		/**
		 * This is a passthrough to the request's
		 * {@link javax.servlet.http.HttpServletRequest#setAttribute(String, Object)}
		 * function. It's just for source prettyness than anything, so rather
		 * than say myActionInfo.request.setAttribute you can say
		 * myActionInfo.setRequestAttribute
		 * 
		 * @param attributeName The attribute to set
		 * @param attributeValue The value to set the attribute to
		 */
		void setRequestAttribute(String attributeName, Object attributeValue)
		{
			request.setAttribute(attributeName, attributeValue);
		}

		/**
		 * This is a passthrough to the request's
		 * {@link javax.servlet.http.HttpServletRequest#getAttribute(String)}
		 * function. It's just for source prettyness than anything, so rather
		 * than say myActionInfo.request.getAttribute you can say
		 * myActionInfo.getRequestAttribute
		 * 
		 * @param attributeName The request attribute to get
		 * @return The value of that request attribute.
		 */
		Object getRequestAttribute(String attributeName)
		{
			return request.getAttribute(attributeName);
		}
		
		ActionForward findForward(String fwdName)
		{
			ActionForward fwd = new ActionForward(mapping.findForward(fwdName));
			
			Util.setRedirectAndParams(fwd, request);
			
			return fwd;
		}

		public ActionForward smartForward(ThreadedUserAction thread, 
				String defaultForward)
		{
			if(thread.isFinished() && thread.getException()==null)
			{
				// Complete success
				
				ActionForward forward = 
					new ActionForward(mapping.findForward(defaultForward));
				Util.setRedirectAndParams(forward, request);
				return forward;
			}
			if(thread.isFinished() /* && thread.getException() != null*/ )
			{
				// Complete failure
				
				request.getSession().setAttribute(Attributes.error.message, 
						thread.getException());
				return mapping.findForward(Mappings.Global.error);
			}
			/* if(!thread.isFinished()) */
			{
				// Incomplete 
				
				return mapping.findForward(Mappings.Global.timeout);
			}
		}
	}

	/**
	 * This function does the behind-the-scenes magic of all the tedious
	 * mucking about mentioned above.
	 * 
	 * @see org.apache.struts.action.Action#execute(org.apache.struts.action.ActionMapping,
	 *      org.apache.struts.action.ActionForm,
	 *      javax.servlet.http.HttpServletRequest,
	 *      javax.servlet.http.HttpServletResponse)
	 */
	public final ActionForward execute(ActionMapping mapping, ActionForm form,
			HttpServletRequest request, HttpServletResponse response)
			throws java.lang.InterruptedException, java.io.UnsupportedEncodingException
	{
		Util.checkForDoneActions(request);

		HttpSession session = request.getSession();

		ThreadInitInfo tii = (ThreadInitInfo)session
				.getAttribute(Attributes.session.threadInfo);
		if(tii == null)
		{
			String o = Util.buildRequestString(request);
			
			//System.out.println("Relogin; forward to " + o);
			
			session.setAttribute(Attributes.login.forwardTo_OPT, o);
			
			ActionForward fwd = new ActionForward(mapping
					.findForward(Mappings.Global.reauthenticate));
			fwd.setRedirect(true);
			return fwd;
		}

		Cache cache = (Cache)session.getServletContext().getAttribute(
				Attributes.servletContext.cache);
		
		tii.setCache(cache);

		if(cache == null || cache.getState()==Cache.UNSET)
		{
			ActionForward fwd = new ActionForward("/pages/UpdateCache.jsp");
			
			String currentURL = Util.buildRequestString(request);
			try
			{
				currentURL = URLEncoder.encode(currentURL, "UTF-8");
			}
			catch (java.io.UnsupportedEncodingException e)
			{
				e.printStackTrace();
				throw e;
			}
			
			String fwdURL = fwd.getPath() + "?forwardTo=" + currentURL;
			fwd.setPath(fwdURL);
			fwd.setRedirect(true);
			
			System.out.println("Cache bad, forwarding to " + fwd.getPath());
			return fwd;
		}

		
		ActionForward fwd = act(new ActionInfo(mapping, form, request, response, tii, cache));
		
		System.out.println("Forward is " + fwd.getPath());
		
		// We look for ".page' at the end
		if(fwd.getPath().matches(".*\\.page$"))
		{
			String tile = fwd.getPath().replaceAll("\\.page$", ".help");
			
			System.out.println("...which means the help tile is " + tile);
			
			request.setAttribute("helpTile", tile);
		}
		
		return fwd;
	}

	/**
	 * This is called automagically when a request is made. Implement this method
	 * when subclassing this to do something.
	 * 
	 * @param info Holds the parameters that were passed to <tt>execute</tt>,
	 *            plus things like the cache and threads info deduced from those
	 *            parameters
	 * @return An object that implements getForward
	 * @throws java.lang.InterruptedException
	 */
	abstract public ActionForward act(ActionInfo info);
}
