/*
 * zVMSMAPIConnector.java
 *
 * Created on June 7, 2004, 8:20 AM
 * Updated Jun 16th - Aug 12th, 2005 - Modified to work with non-RPC based SMAPI server.
 */
package live.smapi;

import java.net.*;
import java.io.*;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.StringTokenizer;

import live.smapi.exceptions.*;
import live.dto.Network;
import live.dto.NameList;

/**
 * SMAPI_Interface is a library that utilizes the z/VM SMAPI to provide the 
 * same feature set as the SMAPI but without the trouble of the confusing 
 * technical details of the SMAPI.  You will not need to check return or
 * reason codes, or worry about setting the correct parameters.  This library
 * provides procedures that reduce the complexity of many SMAPI procedure calls
 * by using sane default values for arguments and reducing the amount of 
 * usually unneeded options available to the programmer.  It also provides 
 * procedures that exactly corespond to the SMAPI calls so that the programmer
 * can still utilize the full power of the SMAPI if neccessary.
 *
 * Note:  Not all functions have been written, and some that have been written
 * do not allow you to make full use of the underlying SMAPI call.  In these cases,
 * the limitations are clearly explained.
 *
 * @author Jason J. Herne (jjherne@us.ibm.com), Emily Kate Celeskey (eceleske@us.ibm.com)
 */
public class SMAPI_Interface
{
    /* SMAPI Function names */
    private static final String NAMELIST_QUERY	    =   "DMSCSNLQ";
    private static final String AUTH_LIST_ADD	    =   "DMSCSXLA";
    private static final String AUTH_LIST_QUERY    =   "DMSCSXLQ";
    private static final String AUTH_LIST_REM	    =   "DMSCSXLR";
    private static final String IMAGE_ACTIV	    =   "DMSCSIAC";
    private static final String IMAGE_CREATE_DM    =   "DVHCSICR";
    private static final String IMAGE_DEACT	    =   "DMSCSIDA";
    private static final String IMAGE_DELETE_DM    =   "DVHCSIDE";
    private static final String IMAGE_DEDICATE     =   "DMSCSVCR";
    private static final String IMAGE_DEDIC_DM	    =   "DVHCSVCR";
    private static final String IMAGE_DEV_RESE	    =   "DMSCSIDR";
    private static final String IMAGE_UNDED	    =   "DMSCSVDE";
    private static final String IMAGE_UNDED_DM	    =   "DMHSSVDE";
    private static final String IMAGE_DISK_CP	    =   "DMSCSDCO";
    private static final String IMAGE_DISK_CP_DM   =   "DVHCSDCO";
    private static final String IMAGE_DISK_CREAT   =   "DMSCSDCR";
    private static final String IMAGE_DISK_CREAT_DM=   "DVHCSDCR";
    private static final String IMAGE_DISK_DEL	    =   "DMSCSDDE";
    private static final String IMAGE_DISK_DEL_DM  =   "DVHCSDDE";
    private static final String IMAGE_DISK_SH	    =   "DMSCSDLC";
    private static final String IMAGE_DISK_SH_DM   =   "DVHCSDLC";
    private static final String IMAGE_DISK_UNSH    =   "DMSCSDLD";
    private static final String IMAGE_DISK_UNSH_DM =   "DVHCSDLD";
    private static final String IMAGE_LOCK_DM	    =   "DVHCSILK";
    private static final String IMAGE_NAME_QU_DM   =   "DVHCSINQ";
    private static final String IMAGE_PSWD_SET_DM  =   "DVHCSIPS";
    private static final String IMAGE_QUERY_DM	    =   "DVHCSIQR";
    private static final String IMAGE_RECY	    =   "DMSCSIRC";
    private static final String IMAGE_REPLACE_DM   =   "DVHCSIRP";
    private static final String IMAGE_STAT_QUER    =   "DMSCSISQ";
    private static final String IMAGE_UNLCK_DM	    =   "DVHCSIUL";
    private static final String IMAGE_VOL_ADD	    =   "DMSCSIVA";
    private static final String IMAGE_VOL_DEL	    =   "DMSCSIVD";
    private static final String IMAGE_VOL_DEF_DM   =   "DVHCSISD";
    private static final String IMAGE_VOL_QUE_DM   =   "DVHCSIVQ";
    private static final String IMAGE_VOL_REM_DM   =   "DVHCSISR";
    private static final String NAME_LIST_ADD	    =   "DMSCSNLA";
    private static final String NAME_LIST_DES	    =   "DMSCSNLD";
    private static final String NAME_LIST_REM	    =   "DMSCSNLR";
    private static final String PROTO_CREA_DM	    =   "DVHCSPCR";
    private static final String PROTO_DEL_DM	    =   "DVHCSPDE";
    private static final String PROTO_NM_QU_DM	    =   "DVHCSPNQ";
    private static final String PROTO_QUE_DM	    =   "DVHCSPQR";
    private static final String PROTO_REPL_DM	    =   "DVHCSPRP";
    private static final String SS_ACC_ADD_DM	    =   "DVHCSSAA";
    private static final String SS_ACC_QUE_DM	    =   "DVHCSSAQ";
    private static final String SS_ACC_REM_DM	    =   "DVHCSSAR";
    private static final String SS_CREATE	    =   "DMSCSSSC";
    private static final String SS_DELETE	    =   "DMSCSSSD";
    private static final String SS_QUERY	    =   "DMSCSSSQ";
    private static final String SS_REPLACE	    =   "DMSCSSSR";
    private static final String STAT_CHANG_ACT_DM  =   "DVHCSVCA";
    private static final String STAT_CHANG_DEA_DM  =   "DVHCSUCD";
    private static final String STAT_CHANG_IMM_DM  =   "DVHCSUCM";
    private static final String VIRT_NET_AD_CR	    =   "DMSCSACR";
    private static final String VIRT_NET_AD_CR_DM  =   "DVHCSACR";
    private static final String VIRT_NET_AD_DE	    =   "DMSCSADE";
    private static final String VIRT_NET_AD_DE_DM  =   "DVHCSADE";
    private static final String VIRT_NET_CON_CR    =   "DMSCSCCR";
    private static final String VIRT_NET_CON_CR_DM =   "DVHCSCCR";
    private static final String VIRT_NET_CON_DE    =   "DMSCSCDE";
    private static final String VIRT_NET_CON_DE_DM =   "DVHCSCDE";
    private static final String VIRT_NET_LAN_CON   =   "DMSCSLCN";
    private static final String VIRT_NET_LAN_CON_DM=   "DVHCSLCN";
    private static final String VIRT_NET_LAN_DS    =   "DMSCSLDC";
    private static final String VIRT_NET_LAN_DS_DM =   "DVHCSLDC";
    private static final String VIRT_NET_LAN_QU    =   "DMSCSLQU";
    private static final String VIRT_NET_VSW_CON   =   "DMSCSVSC";
    private static final String VIRT_NET_VSW_CON_DM=   "DVHCSVSC";
    private static final String VIRT_NET_VSW_DS    =   "DMSCSVSD";
    private static final String VIRT_NET_VSW_DS_DM =   "DVHCSVSD";
    private static final String VIRT_NET_VSW_QUE   =   "DMSCSVSQ";
    private static final String VIRT_NET_VSW_SET   =   "DMSCSVSS";
    private static final String QUERY_ASYNC_OP	    =	"DVHCSQAO";
    
    private static final String NULL_CALL	    =   "SSSTRY";
    private static final String VIRT_NET_VSW_MAKE  =   "SSSVMAKE";
    private static final String ACTIVE_IMAGE_QUERY =   "SSSQACTV";
    private static final String NAME_LIST_Q_ALL    =   "SSSNAMEL";
    private static final String QUERY_VIRTUAL      =   "SSSQVIRT";
    private static final String LOW_FREE_ADDR      =   "SSSFNDAD";
    private static final String FIND_STORAGE       =   "SSSFNDST";
    private static final String QUER_NIC_ADDR      =   "SSSQNTAD";
    private static final String QUER_NET_MEMBERS   =   "SSSQNTME";
    private static final String VIRT_SWIT_CREATE   =   "SSSVMAKE";
    private static final String VSW_DEL	    =	"SSSVDEL";
    private static final String VSW_DET	    =	"SSSVDET";
    //private static final String Q_NSS	    =	"SSSQNSS"; //NO longer needed! (Query virtual used now)
    
    
    private static final byte B_NULL	    = 0x00;
    private static final byte B_LINETERM   = 0x01;
    private static final byte B_SPACE	    = 0x20;
    
    public static final int NL_GUESTADDED =		 0;
    public static final int NL_LISTMADE_GUESTADDED =	 1;
    public static final int NL_LISTMADE_GUESTTHERE =	 2;
    
    private String hostName;
    private int port;
    private String userName;
    private String password;
    
    /**
     * Creates a new SMAPI Interface object with the given parameters.  The userName
     * and password parameter are used to authenticate to the SMAPI server.
     *
     * @param	hostName The host name or IP address  of the z/VM system to connect to.
     * @param	port The port number the SMAPI server is listening on.
     * @param	userName The user name used to authenticate to the SMAPI server.
     * @param	password The password used to authenticate to the SMAPI server.
     */
    public SMAPI_Interface(String hostName, int port, String userName, String password)
    {
	this.hostName = hostName;
	this.port = port;
	this.userName = userName;
	this.password = password;
    }
    
    /** 
     * Executes a simple SMAPI call that does nothing other than authenticate with 
     * the SMAPI server.  This serves no purpose other than to check and see if
     * the username and password supplied are a valid combination.
     */
    public void nullRequest()
	throws SMAPIException
    {
	// The comments in this function serve as a guide to help you write a 
	// wrapper function for a new SMAPI call.  to save space, they are not 
	// repeated in all of the other functions for which they apply.
    
	// The first step in writting a new SMAPI wrapper function is to create
	// a constant that contains the name of the SMAPI function you are going
	// to call.  The list of constants can be found at the very top of this
	// class.  Add your new constant to this list.  In this particular case
	// we added the NULL_CALL constant.

	// Call connectRequest to send the request to the SMAPI server.  
	// connectRequest takes as argument the name of the call we are making
	// and the space delimited list of arguments to the SMAPI call being
	// made.  In the event that there are no arguments, a simple null value
	// should be passed.
	LinkedList tokens = connectRequest(NULL_CALL, null);
	
	// connectRequest will always return a linkedList of Strings.  The first 
	// element of the list will always be the SMAPI RC value of the call that
	// was just executed.  The second element of the list will always be the
	// SMAPI RS value of the call that was just executed.  We simply remove 
	// the RC and RS from the list and save them as integers for later 
	// comparison.
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	// If you are looking for very specific RC/RS values in
	// order to decide what happens next, you would do that now.  You should note
	// that this is not the general case.  Generally rc/rs values only return 
	// non-zero in the event of an error.  If your function needs to catch specific
	// returns codes that do not mean an error has occurred, this is where to
	// do that.  For an example of this, see the NamelistAdd function.
	
	// Now we need to check and see if an error has occurred.  Most SMAPI calls
	// will simply be able to use the errorCheck function.  This function takes
	// the following parameters: description of your call, rc, and rs values.
	// errorCheck will compare the rc/rs values against its list of errors and
	// throw the appropriate exception.  Keep in mind, however, that some SMAPI 
	// calls give special meaning to rc/rs pairs that also have a general meaning.
	// Example: rc=400/rs=16 generally means Image Def. can't be deleted, but
	// if your call uses 400/16 to represent a different error, you will need          
	// to check for this BEFORE you call errorCheck or else the wrong exception
	// will be thrown.
	errorCheck("SMAPI Null",  rc,  rs);
	
	// At this point, we know that an error has not occurred.  In the case of 
	// the  nullRequest, we can return.  However, if your SMAPI call was
	// supposed to return some data for you (Example: NamelistQuery") you'll
	// need to interpret and/or return that data as you see fit.
	// Any SMAPI call response data was actually returned by the call to 
	// connectRequest and can be found in the LinkedList that it returned.
	// As a matter of fact, because we removed the rc and rs values from that
	// list, the data that the SMAPI call was supposed to return to you is
	// all that is left in that list of Strings, titled tokens in this case.
	
	// In many cases, you might simply return the tokens list, this is what 
	// is done in imageNameQuery because a list of Strings is a sensible way
	// to represent a whole bunch of guest names.
	
	// NOTE:  There is also a connectRequestRaw which will send the parameters
	// String EXACTLY as you pass it in.  No translation or interpretation will
	// be done.  No extra parameter terminators will be sent.  Use this if your
	// call requires special handling of paramters that is no automatically
	// handled by connectRequest.
    }
    
    /**
     * Corresponds to the SMAPI procedure NAME_LIST_QUERY.  Queries the 
     * contents of a name list.
     *
     * @param listName The name of the namelist to query.
     * @return LinkedList A List of Strings, each one is a name in the namelist querried.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList nameListQuery(String listName)
	throws SMAPIException
    {
	    LinkedList tokens = connectRequest(NAMELIST_QUERY, listName);
	    
	    int rc=Integer.parseInt((String)tokens.removeFirst());
	    int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	    if (rc != 0) errorCheck("Name list query",  rc,  rs);
	 
	    return tokens;
    }
    
    /**
     * Corresponds to the SMAPI procedure NAME_LIST_ADD.
     * Creates/adds a new name to a name list.  This function will add a name to
     * an existing name list.  If the name list specified does not exist, a new 
     * list will be created and nameToAdd will be added to the new list. Returns
     * an integer.
     * 
     * The return value's meaning is as follows:
     *  0 - the guest was added to the (existing) name list
     *  1 - the guest was added to a new name list
     *  2 - the guest was already a member of the name list, and no change was made
     *
     * @param listName Name list to which an entry will be added.
     * @param nameToAdd Name to add to the name list.
     * @return 0, 1, or 2 corresponding to addition, creation, or no change as described above.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int nameListAdd(String listName, String nameToAdd)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(NAME_LIST_ADD, listName + " " + nameToAdd);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if (rc == 0 && rs == 0) return 0; //list exists, guest was added
	else if (rc == 0 && rs == 12) return 1; //list was created, guest was added
	else if (rc == 0 && rs == 36) return 2; //list exists, guest already there
	else errorCheck("Namelist Add",  rc,  rs);
	
	return -1; // we Never get here, but Java makes us do this.
    }

    /**
     * Corresponds to the SMAPI procedure NAME_LIST_DESTROY.  Deletes a name 
     * list.
     *
     * @param listName The name of the list to destroy.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void nameListDestroy(String listName)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(NAME_LIST_DES, listName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Namelist Destroy",  rc,  rs);
    }

    /**
     * Corresponds to the SMAPI procedure NAME_LIST_REMOVE.
     * Delete/remove an entry from a name list.  If the removal causes the 
     * list to be empty, the list will be deleted.
     *
     * The return value's meaning is as follows:
     *  0 - the guest was removed from the name list
     *  1 - the guest was removed, list was destroyed
     *  2 - the guest was not in the list
     *
     * @param strListName name list from which to remove an entry.
     * @param strNameToRemove Name to remove from the name list.
     * @return int Code to tell how the removal went.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int nameListRemove(String strListName, String strNameToRemove)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(NAME_LIST_REM, strListName + " " + strNameToRemove);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	if (rc == 0 && rs == 0) return 0; //list exists, guest was removed
	else if (rc == 0 && rs == 16) return 1; //list removed, guest removed
	else if (rc == 0 && rs == 32) return 2; //guest was not in list
	errorCheck("Namelist remove",  rc,  rs);
	
	return -1; //Never gets here.
    }  

    /**
     * Corresponds to the SMAPI procedure QUERY_ASYNCHRONOUS_OPERATION_1.  This 
     * call is used by smapilib to wait for asynchronous operations to complete.
     * This is not something that can currently be used by the end user because
     * smapilib automatically calls this function which will return as soon as
     * the asyncronous operation is complete.
     *
     * @param asyncOperationID ID of the asynchronous operation.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    private void asynchronousCall(String asyncOperationID)
        throws SMAPIException
    {
        int timer=0;

	while(true)
	{
	    /* The JUNK param is not used my the Q ASYNC OPS script, but it needs to be there, dont know why */
	    LinkedList tokens = connectRequest(QUERY_ASYNC_OP,  "JUNK " + asyncOperationID);

	    int rc=Integer.parseInt((String)tokens.removeFirst());
	    int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	    //Check results and return only if operation is finished.
	    //Tell user in the case of an error, and loop if we
	    //are still waiting.
	    
	    if (rc == 0 && rs == 100) return; //Call is complete.
	    else if(rc == 0 && rs == 108) throw new SMAPIException("Async call failed.  No further information is available.");
	    
	    else if (rc == 0 && rs == 104) //Still waiting
	    {
		//5s * 48  = 4m * 3 = 12m
		//Only allow 4 minutes for async call to return.
		if (timer == 48*3) throw new SMAPIException("Async error: Timed out waiting for asynchronous call to return.");

		timer++;
		 
		//Sleep for 5 seconds and try again
		try{ Thread.sleep(5000); }
		catch (java.lang.InterruptedException e) { throw new SMAPIException("Async error: Thread Interrupted."); }
	    }
	    else //Some other error
	    {
		errorCheck("Asynchronous Operation Call:", rc, rs);
	    }
	}
    }
   
    /**
     * Corresponds to the SMAPI procedure PROTOTYPE_NAME_QUERY.  Used to 
     * obtain a list of all prototypes that exist.
     *
     * @return Linked list containing Strings corresponding to the prototype names.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList prototypeNameQuery()
        throws SMAPIException
    {
        //Not really sure what is needed as parameter here...
	LinkedList tokens = connectRequest(PROTO_NM_QU_DM, "yo");
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	 
        errorCheck("Prototype name query",  rc,  rs);
	
	return tokens; 
    }
   
    /**
     * Corresponds to the SMAPI procedure PROTOTYPE_CREATE.  Creates a new
     * prototype based on the provided statements.  A prototype is a pseudo 
     * directory entry that can be used as a model to create new guests.
     *
     * @param name The name of throws prototype to create
     * @param linesList A LinkedList of strings.  Each string is a line/entry in the new prototype.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void prototypeCreate(String name, LinkedList linesList)
        throws SMAPIException
    {
	// Need to send the name, then lines.
	// The lines need to be 0x01 terminated.
	StringBuffer rawParms = new StringBuffer("");
	
	rawParms.append(name);
	rawParms.append((char)B_NULL);
	
	ListIterator itr = linesList.listIterator();
	while(itr.hasNext())
	{
	    rawParms.append((String)itr.next());
	    rawParms.append((char)B_LINETERM);
	}
	rawParms.append((char)B_NULL);
	
	LinkedList tokens = connectRequestRAW(PROTO_CREA_DM, rawParms.toString());
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Prototype create",  rc,  rs);
    }
      
    /**
     * Corresponds to the SMAPI procedure PROTOTYPE_DELETE.  Deletes an 
     * existing prototype.
     *
     * @param prototypeName Name of the prototype to delete.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void prototypeDelete(String prototypeName)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(PROTO_DEL_DM, prototypeName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Prototype delete",  rc,  rs);
    }
   
    /**
     * Corresponds to the SMAPI procedure PROTOTYPE_QUERY.  Querries a 
     * prototype and returns its contents lines/entries are newline separated.
     *
     * @param prototypeName Name of the prototype to return.
     * @return Contents of the prototype querried.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList prototypeQuery(String prototypeName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(PROTO_QUE_DM, prototypeName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Prototype query",  rc,  rs);
            
	return tokens; 
    }
    
    /**
     * Corresponds to the SMAPI procedure IMAGE_NAME_QUERY.  Obtains a 
     * complete list of guests defined on the system.
     *
     * @return LinkedList List containing all defined guests.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList imageNameQuery()
        throws SMAPIException 
    {
	LinkedList tokens = connectRequest(IMAGE_NAME_QU_DM, userName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	errorCheck("Image name query",  rc,  rs);
            
	return tokens;   
    }

    
    /**
     * Corresponds to the SMAPI procedure IMAGE_CREATE.  Creates a new guest.  
     * The account number parameter to the original SMAPI call has ben ignored.
     *
     * @param guestName Name of the new guest.
     * @param prototypeName Name of the prototype this guest will be modeled after.
     * @param password Password for this guest.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageCreate(String guestName, String prototypeName, String password)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_CREATE_DM, guestName + " " + prototypeName + " " + password + " &");
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	errorCheck("Guest create",  rc,  rs);
    }

    /**
     * Corresponds to the SMAPI procedure IMAGE_REPLACE.  Replaces a guests 
     * directory entry with the specefied lines.
     *
     * @param imageName name of the guest to replace.
     * @param imageDefLines Array containing strings that represent the lines of the new directory entry for this guest.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageReplace(String imageName, LinkedList imageDefLines)
        throws SMAPIException
    {
        // Need to send the name, then lines.
	// The lines need to be 0x01 terminated.
	StringBuffer rawParms = new StringBuffer("");
	
	rawParms.append(imageName);
	rawParms.append((char)B_NULL);
	
	ListIterator itr = imageDefLines.listIterator();
	while(itr.hasNext())
	{
	    rawParms.append((String)itr.next());
	    rawParms.append((char)B_LINETERM);
	}
	rawParms.append((char)B_NULL);
	
	LinkedList tokens = connectRequestRAW(IMAGE_REPLACE_DM, rawParms.toString());
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Image replace",  rc,  rs);
    }

    /**
     * Corresponds to the SMAPI procedure IMAGE_DELETE.  Deletes a guest from
     * the system.  
     *
     * @param guestName name of the guest to delete.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDelete(String guestName)
        throws SMAPIException
    {
        
      	LinkedList tokens = connectRequest(IMAGE_DELETE_DM, guestName + " " + "2");
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	errorCheck("Image delete",  rc,  rs);
    }
 
    /**
     * Corresponds to the SMAPI procedure IMAGE_QUERY.  Querries a guest and returns the
     * statements in the guests directory entry as a newline separated string.
     *
     * @param guestName The name of the guest to query.
     * @return String containing this guests directory entry.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList imageQuery(String guestName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_QUERY_DM, guestName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Image query",  rc,  rs);

	//Remove line resembling this:
	// *DVHOPT LNK0 LOG1 RCM1 SMS0 NPW1 LNGAMENG PWC20041028 CRC?
	tokens.removeLast();
	
	//Remove the numbers at the end of all the remaining lines.
	// Ex: USER $ALLOC$ NOLOG 16M 32M                               07211422
	
	//Were going to remove Strings from the tokens list and add them (modified)
	//to a new list.
	LinkedList linesToReturn = new LinkedList();
	while(!tokens.isEmpty())
	{
	    String curLine = (String)tokens.removeFirst();
	    StringBuffer newLine = new StringBuffer("");
	    
	    //Remove the weird number from the end of the current line.
	    StringTokenizer strTok = new StringTokenizer(curLine);
	    
	    while(strTok.countTokens() > 1)
	    {
		newLine.append(strTok.nextToken());
		newLine.append(" ");
	    }
	    
	    //System.out.println("-");
	    linesToReturn.addLast(newLine.toString());
	}
	
	
	return linesToReturn; 	//Each token will be a line.
    }
    
    /**
     * Corresponds to the SMAPI procedure IMAGE_PASSWORD_SET.  Changes the 
     * password for a guest.
     *
     * @param guestName  Name of the guest to set the password for.
     * @param password The new password for the guest.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageSetPassword(String guestName, String password)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_PSWD_SET_DM, guestName + " " + password);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("ImageSetPassword",  rc,  rs);
    }
   
    /**
     * Corresponds to the SMAPI procedure IMAGE_LOCK.  Locks a guest so that
     * changes cannot be made to it.  Any changes to the guest or device will
     * result in an "Image locked" error.  This wraper does not allow you to
     * locak individual devices, just the guest as a whole.
     *
     * @param guestName The guest to lock, or the guest who owns the device you wish to lock.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageLock(String guestName)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(IMAGE_LOCK_DM, guestName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Image lock",  rc,  rs);
    }       

    /**
     * Corresponds to the SMAPI procedure IMAGE_UNLOCK.  unlocks a locked 
     * guest.
     *
     * @param guestName The guest to unlock
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageUnlock(String guestName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_UNLCK_DM, guestName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	errorCheck("Image unlock",  rc,  rs);
    }   

    /**
     * Corresponds to the SMAPI procedure IMAGE_ACTIVATE.  Logs on a guest.
     * Unlike the underlying SMAPI call, we do not allow the use of namelists
     * here.
     * @param guestName The guest (or name list) to activate (log on).
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageActivate(String guestName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_ACTIV, guestName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image activate",  rc,  rs);
    }
    
    /**
     * Corresponds to the SMAPI procedure IMAGE_DEACTIVATE.  logs a guest off
     * of the system.  The underlying SMAPI call allws the caller to specify a
     * time limit before the guest is forced off the system, we always force
     * immediately to ensure quick deactivation.
     *
     * @param guestName The name of the guest to log off.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDeactivate(String guestName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(IMAGE_DEACT, guestName + " IMMED");
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image deactivate",  rc,  rs);
    }

    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_LAN_QUERY.  Querries
     * all guest LANs and VSwitches on the system and returns the information as 
     * a LinkedList Network objects.  The output Strings are equivalent to what you 
     * would get if you issed the CP command "QUERY LAN ALL".  The underlying SMAPI
     * call allows the caller to specific any LAN name, we only support querying
     * all LANs.
     *
     * @return LinkedList of network objects.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList virtualLanQuery()
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(VIRT_NET_LAN_QU, "ALL");
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Virtual LAN query",  rc,  rs);
	
	LinkedList networks = parseNetworks(tokens);  
	return networks;       
    }
     
    /**
     * Parses output from the query networks SMAPI call.
     * Reads the output from the SMAPI call VIRTUAL_NETWORK_LAN_QUERY and
     * returns a list of Network objects, one for each network present in the
     * output.
     * @param tokens The output from VIRTUAL_NETWORK_LAN_QUERY
     * @return LinkedList List of network objects.
     */
    private LinkedList parseNetworks(LinkedList tokens)
	throws SMAPIException
    {
	LinkedList networks = new LinkedList();
	
	//While there are lines left.
	while(tokens.size() != 0)
	{
	    if( ((String)tokens.get(0)).startsWith("LAN") ) 
		networks.add(parseGLan(tokens));
	    else if( ((String)tokens.get(0)).startsWith("VSWITCH") ) 
		networks.add(parseVSwitch(tokens));
	    else
		tokens.removeFirst();
	}
	
	return networks;
    }
    
    /**
     * Parses Guest LAN output from the query networks SMAPI call.
     * Given a section of output from VIRTUAL_NETWORK_LAN_QUERY, remove the lines
     * corresponding to a Guest LAN existing at the top of the output from the list,
     * parse those lines to determine information about the guest LAN, and 
     * return a Network object repreenting that guest LAN.
     *
     * @param lines Output from VIRTUAL_NETWORK_LAN_QUERY
     * @return Network object representing a LAN.
     */
    private Network parseGLan(LinkedList lines)
	throws SMAPIException
    {
	/* FORMAT
	
        LAN SYSTEM GLANA        Type: HIPERS  Connected: 0    Maxconn: INFINITE
	      PERSISTENT  UNRESTRICTED  IP        MFS: 16384      Accounting: OFF
	*/
	if(lines.size() < 2) throw new SMAPIException("Error parsing GLAN:  Not enough lines.");
	String line1 = (String)lines.removeFirst();
	String line2 = (String)lines.removeFirst();
	
	StringTokenizer strTok = new StringTokenizer(line1);
        if( strTok.countTokens() < 9)
	    throw new SMAPIException("Error parsing GLAN:  Not enough tokens in line 1.");
	strTok.nextToken(); //Eat 'LAN'
	String glanOwner = strTok.nextToken();
	String glanName = strTok.nextToken();
	strTok.nextToken(); //Eat 'Type:'
	String glanType = strTok.nextToken();
	strTok.nextToken(); //Eat 'Connected:'
	String glanNumConnected = strTok.nextToken();
	strTok.nextToken(); //Eat 'Maxconn:'
	String glanMaxCon= strTok.nextToken();
	
	strTok = new StringTokenizer(line2);
	if( strTok.countTokens() < 3)
	    throw new SMAPIException("Error parsing GLAN:  Not enough tokens in line 2.");
	strTok.nextToken(); //Eat 'PERSISTENT'
        String glanRestrict = strTok.nextToken();
	strTok.nextToken(); //Eat 'IP' or 'ETHERNET'
       
	//Fill in the network object with the data we read.
	Network glan = new Network(glanName, Network.NETWORK_GLAN);
	glan.owner = glanOwner;
	glan.type = glanType;
	glan.numConnected = Integer.parseInt(glanNumConnected);
	
	if(glanMaxCon.equals("INFINITE")) glan.maxConnected = -1;
	else glan.maxConnected = Integer.parseInt(glanMaxCon);
	
	glan.restricted = glanRestrict.equals("RESTRICTED");
	
	return glan;   
   }
   
    /**
     * Parses VSwitch output from the query networks SMAPI call.
     * Given a section of output from VIRTUAL_NETWORK_LAN_QUERY, remove the lines
     * corresponding to a VSwitch existing at the top of the output from the list,
     * parse those lines to determine information about the VSwitch, and 
     * return a Network object representing that VSwitch.
     *
     * @param lines Output from VIRTUAL_NETWORK_LAN_QUERY
     * @return Network object representing a LAN.
     */
    private Network parseVSwitch(LinkedList lines)
	throws SMAPIException
    {
	/* FORMAT
	
	    VSWITCH SYSTEM VSW1     Type: VSWITCH Connected: 0    Maxconn: INFINITE
	      PERSISTENT  RESTRICTED    NONROUTER                 Accounting: OFF
	      VLAN Unaware
	      State: Defined
	      IPTimeout: 5         QueueStorage: 8
optional      Portname: ASD        RDEV: 0100 Controller: NONE     Error: No RDEV
optional      Portname: QWE        RDEV: 0200 Controller: NONE     Error: No RDEV
optional      Portname: FGH        RDEV: 0300 Controller: NONE     Error: No RDEV

	*/
	if(lines.size() < 5) 
	{
	    throw new SMAPIException("Error parsing VSwitch:  Not enough lines.");
	}
	String line1 = (String)lines.removeFirst();
	String line2 = (String)lines.removeFirst();
	/*String line3 = (String)*/ lines.removeFirst(); // Skip line 3
	/*String line4 = (String)*/ lines.removeFirst(); // Skip line 4
	String line5 = (String)lines.removeFirst();
	
	StringTokenizer strTok = new StringTokenizer(line1);
        if( strTok.countTokens() < 9)
	    throw new SMAPIException("Error parsing VSwitch:  Not enough tokens in line 1.");
	strTok.nextToken(); //Eat 'VSWITCH'
	String vswOwner = strTok.nextToken();
	String vswName = strTok.nextToken();
	strTok.nextToken(); //Eat 'Type:'
	strTok.nextToken(); //Eat 'VSWITCH'
	strTok.nextToken(); //Eat 'Connected:'
	String vswNumConnected = strTok.nextToken();
	strTok.nextToken(); //Eat 'Maxconn:'
	String vswMaxCon= strTok.nextToken();
	
	strTok = new StringTokenizer(line2);
	if( strTok.countTokens() < 3)
	    throw new SMAPIException("Error parsing GLAN:  Not enough tokens in line 1.");
	strTok.nextToken(); //Eat 'PERSISTENT'
        String vswRestrict = strTok.nextToken();
	String vswRoute = strTok.nextToken(); 
       
	strTok = new StringTokenizer(line5); //We dont use line3 or 4
	if( strTok.countTokens() < 4)
	    throw new SMAPIException("Error parsing GLAN:  Not enough tokens in line 1.");
	strTok.nextToken(); //Eat 'IPTimeout:'
        String vswIPTimeout = strTok.nextToken();
	strTok.nextToken(); //Eat 'QueueStorage:'
	String vswQS = strTok.nextToken(); 
	
	// Now, we need to possible parse some real device lines, there can be
	// 0 to 3 of them.
	String vswDevice1=null, vswDevice2=null, vswDevice3=null;
	String vswPort1=null, vswPort2=null, vswPort3=null;
	
	// Real device #1
	if (lines.size() > 0 && ((String)lines.getFirst()).startsWith("  Portname:"))
	{
	    strTok = new StringTokenizer((String)lines.removeFirst()); 
	    strTok.nextToken(); //Eat 'Portname:'
	    vswPort1 = strTok.nextToken();
	    strTok.nextToken(); //Eat 'RDEV::'
	    vswDevice1 = strTok.nextToken();
	}
	
	// Real device #2
	if (lines.size() > 0 && ((String)lines.getFirst()).startsWith("  Portname:"))
	{
	    strTok = new StringTokenizer((String)lines.removeFirst()); 
	    strTok.nextToken(); //Eat 'Portname:'
	    vswPort2 = strTok.nextToken();
	    strTok.nextToken(); //Eat 'RDEV::'
	    vswDevice2 = strTok.nextToken();
	}
	
	// Real device #3
	if (lines.size() > 0 && ((String)lines.getFirst()).startsWith("  Portname:"))
	{
	    strTok = new StringTokenizer((String)lines.removeFirst()); 
	    strTok.nextToken(); //Eat 'Portname:'
	    vswPort3 = strTok.nextToken();
	    strTok.nextToken(); //Eat 'RDEV::'
	    vswDevice3 = strTok.nextToken();
	}

	//Fill in the network object with the data we read.
	Network vsw = new Network(vswName, Network.NETWORK_VSWITCH);
	vsw.owner = vswOwner;
	vsw.numConnected = Integer.parseInt(vswNumConnected);
	
	if(vswMaxCon.equals("INFINITE")) vsw.maxConnected = -1;
	else vsw.maxConnected = Integer.parseInt(vswMaxCon);
	
	vsw.restricted = vswRestrict.equals("RESTRICTED");
	vsw.priRouter = vswRoute.equals("PRIROUTER");
	vsw.ipTimeout = Integer.parseInt(vswIPTimeout);
	vsw.queueStorLim = Integer.parseInt(vswQS);
	vsw.rdev1 = vswDevice1;
	vsw.rdev2 = vswDevice2;
	vsw.rdev3 = vswDevice3;
	
	if (vswPort1 != null && !vswPort1.equals("UNASSIGNED")) vsw.portname1 = vswPort1;
	if (vswPort2 != null && !vswPort2.equals("UNASSIGNED")) vsw.portname2 = vswPort2;
	if (vswPort3 != null && !vswPort3.equals("UNASSIGNED")) vsw.portname3 = vswPort3;
	
	return vsw;   
    }
    
    /**
     * Creates a new minidisk for a guest
     *
     * @param strGuestName Guest to create minidisk for
     * @param strVirtualDeviceAddress Virtual address to create minidisk at
     * @param strAllocationType How to allocate minidisk (T, V, or AUTO)
     * @param strAllocUnitSize What units to allocate the minidisk in (cylinders or blocks)
     * @param strDiskSize How large of a disk to create
     * @param strDiskMode What type of access to have for the minidisk (e.g. RR or RW)
     * @param strReadPW ReadPW for the disk
     * @param strWritePW WritePW for the disk
     * @param strMultiPW MultiPW for the disk
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskCreate(String strGuestName, String strVirtualDeviceAddress, String strAllocationType, String strAllocUnitSize, String strDiskSize, String strDiskMode, String strFormat, String strReadPW, String strWritePW, String strMultiPW)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "3";		// imageInstance = ALL
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 = "3390";	    	//Device Type
	String parm08 = "AUTOG";	//Allocation type
	String parm09 =	"LVDASD";	//Allocation area name
	String parm10 =	"CYLINDERS";	//Alloc unit size
	if (!strAllocUnitSize.equals(" "))
	    parm10 = strAllocUnitSize;
	String parm11    = strDiskSize;	//Image Disk Size
	if (strAllocationType.equals("T-DISK")) 
	{
	    parm08 = "T-DISK"; 	//AllocType
	    parm05 = "2"; //imageInstance = STATIC
	    parm10 = "BLK4096";
	}
	else if (strAllocationType.equals("V-DISK"))
	{
	    parm10 = "BLK0512";
	    parm08 = "V-DISK";		//AllocType
	    parm07 = "FB-512";		//Device type
	}
	    
	String parm12 =	strDiskMode;	//Disk mode
	String parm13 =	strFormat;	//Disk formatting
	String parm14 =	"&";		//Disk Label
	String parm15 =	"&";	//Read pw
	if (!strReadPW.equals(" "))
	    parm15 = strReadPW;
	String parm16 =	"&";	//Write pw
	if(!strWritePW.equals(" "))
	    parm16 = strWritePW;
	String parm17 =	"&";	//Multi pw
	if(!strMultiPW.equals(" "))
	    parm17 = strMultiPW;
	LinkedList tokens = connectRequest(IMAGE_DISK_CREAT, parm04 + " " + parm05 + " " + parm06 + " " +
							     parm07 + " " + parm08 + " " + parm09 + " " +
							     parm10 + " " + parm11 + " " + parm12 + " " +
							     parm13 + " " + parm14 + " " + parm15 + " " +
							     parm16 + " " + parm17);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Create",  rc,  rs);   
    }
    
    /**
     * Creates a new minidisk for a guest's directory entry
     *
     * @param strGuestName Guest to create minidisk for
     * @param strVirtualDeviceAddress Virtual address to create minidisk at
     * @param strAllocationType How to allocate minidisk (T, V, or AUTO)
     * @param strAllocUnitSize What units to allocate the minidisk in (cylinders or blocks)
     * @param strDiskSize How large of a disk to create
     * @param strDiskMode What type of access to have for the minidisk (e.g. RR or RW)
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskCreateDM(String strGuestName, String strVirtualDeviceAddress, String strAllocationType, String strAllocUnitSize, String strDiskSize, String strDiskMode, String strFormat, String strReadPW, String strWritePW, String strMultiPW)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "3";		// imageInstance = ALL
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 = "3390";		//Device Type
	String parm08 = "AUTOG";	//Allocation type
	String parm09 =	"LVDASD";	//Allocation area name
	String parm10 =	"CYLINDERS";	//Alloc unit size
	if (!strAllocUnitSize.equals(" "))
	    parm10 = strAllocUnitSize;
	String parm11    = strDiskSize;	//Image Disk Size
	if (strAllocationType.equals("T-DISK")) 
	{
	    parm08 = "T-DISK"; 	//AllocType
	    parm05 = "2"; //imageInstance = STATIC
	    parm10 = "BLK4096";
	}
	else if (strAllocationType.equals("V-DISK"))
	{
	    parm10 = "BLK0512";
	    parm08 = "V-DISK";		//AllocType
	    parm07 = "FB-512";		//Device type
	}
	    
	String parm12 =	strDiskMode;	//Disk mode
	String parm13 =	strFormat;	//Disk formatting
	String parm14 =	"&";		//Disk Label
	String parm15 =	"&";	//Read pw
	if (!strReadPW.equals(" "))
	    parm15 = strReadPW;
	String parm16 =	"&";	//Write pw
	if(!strWritePW.equals(" "))
	    parm16 = strWritePW;
	String parm17 =	"&";	//Multi pw
	if(!strMultiPW.equals(" "))
	    parm17 = strMultiPW;
	LinkedList tokens = connectRequest(IMAGE_DISK_CREAT_DM, parm04 + " " + parm05 + " " + parm06 + " " +
							     parm07 + " " + parm08 + " " + parm09 + " " +
							     parm10 + " " + parm11 + " " + parm12 + " " +
							     parm13 + " " + parm14 + " " + parm15 + " " +
							     parm16 + " " + parm17);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Create DM",  rc,  rs);   
    }

    /**
     * This function deletes a minidisk from a guest
     *
     * @param strGuestName Name of guest to delete disk from
     * @param strVirtualDeviceAddress Virtual Device of disk to delete
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskDelete(String strGuestName, String strVirtualDeviceAddress)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "3";		// imageInstance = ALL
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 =	"0";	//Data security erase - default option
	LinkedList tokens = connectRequest(IMAGE_DISK_DEL, parm04 + " " + parm05 + " " + parm06 + " " + parm07);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Delete",  rc,  rs);   
    }

    /**
     * This function deletes a minidisk from a guest's directory entry
     *
     * @param strGuestName Name of guest to delete disk from
     * @param strVirtualDeviceAddress Virtual Device of disk to delete
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskDeleteDM(String strGuestName, String strVirtualDeviceAddress)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "3";		// imageInstance = ALL
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 =	"0";	//Data security erase - default option
	LinkedList tokens = connectRequest(IMAGE_DISK_DEL_DM, parm04 + " " + parm05 + " " + parm06 + " " + parm07);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Delete DM",  rc,  rs);   
    }

    // THIS FUNCTION HAS TO WORK AROUND A DIRMAINT BUG, PLEASE READ COMMENTS CAREFULLY
    /**
     * Copies a minidisk from one guest to another
     *
     * @param strGuestName Guest to copy minidisk to
     * @param strVirtualDeviceAddress Virtual address to copy minidisk to
     * @param strSourceImageName Guest to copy minidisk from
     * @param strSourceDeviceAddress Virtual address to copy minidisk from
     * @param strAllocationType How to allocate minidisk (AUTO G, R, or V)
     * @param strDiskMode What type of access to have for the minidisk (e.g. RR or RW)
     * @param strReadPW Read access password
     * @param strWritePW Write access password
     * @param strMultiPW Multi access password
     * @return int 0 if the operation has returned without a failure, 1 if the DirMaint bug has shown up and we cannot determine if the call completed successfully (see comments)
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int imageDiskCopy(String strGuestName, String strVirtualDeviceAddress, String strSourceImageName, String strSourceDeviceAddress, String strAllocationType, String strDiskMode, String strReadPW, String strWritePW, String strMultiPW)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "3";		// imageInstance = ALL
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 =	strSourceImageName; //Source
	String parm08 = strSourceDeviceAddress; //Source address
	String parm09 =	strAllocationType; //Allocation type
	String parm95 = "LVDASD";
	String parm10 =	strDiskMode;	//Disk mode
	String parm11 =	"&";	//Read pw
	if (!strReadPW.equals(" "))
	    parm11 = strReadPW;
	String parm12 =	"&";	//Write pw
	if(!strWritePW.equals(" "))
	    parm12 = strWritePW;
	String parm13 =	"&";	//Multi pw
	if(!strMultiPW.equals(" "))
	    parm13 = strMultiPW;
	LinkedList tokens = connectRequest(IMAGE_DISK_CP, parm04 + " " + parm05 + " " + parm06 + " " +
							     parm07 + " " + parm08 + " " + parm09 + " " + parm95 + " " +
							     parm10 + " " + parm11 + " " + parm12 + " " +
							     parm13);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	  
	//return 0 if the operation was a success
	int retval = 0;
	
	/*if DirMaint has returned with an error code that indicates that a CP command failed
	 *  set retval to 1 (we will use this in the ThreadedZVMTasks layer to tell the end user
	 *  that we cannot determine what happened with this task)
	 * This is a DirMaint bug that will always return that the CP copy command failed, when
	 *  in reality only fast copy failed and it may still be copying the disk.  There is no way 
	 *  to check whether this task has completed successfully other than to check back later.
	 */
	if (rc == 596 && rs == 3212)
	    retval = 1;
	/*if DirMaint has not returned with a CP command error code, let the error check run as
	 * normal, throwing any exceptions necessary
	 */
	else
	    errorCheck("Image Disk Copy",  rc,  rs);   
	
	return retval;
    }

    // THIS FUNCTION HAS TO WORK AROUND A DIRMAINT BUG, PLEASE READ COMMENTS CAREFULLY    
    /**
     * Copies a minidisk from one guest's directory to another guest's directory
     *
     * @param strGuestName Guest to copy minidisk to
     * @param strVirtualDeviceAddress Virtual address to copy minidisk to
     * @param strSourceImageName Guest to copy minidisk from
     * @param strSourceDeviceAddress Virtual address to copy minidisk from
     * @param strAllocationType How to allocate minidisk (AUTO G, R, or V)
     * @param strDiskMode What type of access to have for the minidisk (e.g. RR or RW)
     * @param strReadPW Read access password
     * @param strWritePW Write access password
     * @param strMultiPW Multi access password
     * @return int 0 if the operation has returned without a failure, 1 if the DirMaint bug has shown up and we cannot determine if the call completed successfully (see comments)
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int imageDiskCopyDM(String strGuestName, String strVirtualDeviceAddress, String strSourceImageName, String strSourceDeviceAddress, String strAllocationType, String strDiskMode, String strReadPW, String strWritePW, String strMultiPW)
        throws SMAPIException
    {
	String parm04 = strGuestName;	// Guest
	String parm05 = "2";		// imageInstance = 
	String parm06 = strVirtualDeviceAddress;		// Device Address
	String parm07 =	strSourceImageName; //Source
	String parm08 = strSourceDeviceAddress; //Source address
	String parm09 =	strAllocationType; //Allocation type
	String parm95 = "LVDASD";
	String parm10 =	strDiskMode;	//Disk mode
	String parm11 =	"&";	//Read pw
	if (!strReadPW.equals(" "))
	    parm11 = strReadPW;
	String parm12 =	"&";	//Write pw
	if(!strWritePW.equals(" "))
	    parm12 = strWritePW;
	String parm13 =	"&";	//Multi pw
	if(!strMultiPW.equals(" "))
	    parm13 = "&";
	LinkedList tokens = connectRequest(IMAGE_DISK_CP_DM, parm04 + " " + parm05 + " " + parm06 + " " + parm07 + " " + parm08 + " " + parm09 + " " + parm95 + " " + parm10 + " " + parm11 + " " + parm12 + " " + parm13);
   
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	//return 0 if the operation was a success
	int retval = 0;
	
	/*if DirMaint has returned with an error code that indicates that a CP command failed
	 *  set retval to 1 (we will use this in the ThreadedZVMTasks layer to tell the end user
	 *  that we cannot determine what happened with this task)
	 * This is a DirMaint bug that will always return that the CP copy command failed, when
	 *  in reality only fast copy failed and it may still be copying the disk.  There is no way 
	 *  to check whether this task has completed successfully other than to check back later.
	 */
	if (rc == 596 && rs == 3212)
	    retval = 1;
	/*if DirMaint has not returned with a CP command error code, let the error check run as
	 * normal, throwing any exceptions necessary
	 */
	else
	    errorCheck("Image Disk Copy DM",  rc,  rs);   
	
	return retval;
    }
    /**
     * Corresponds to the SMAPI procedure IMAGE_DISK_SHARE.  Creates a link to
     * a guests virtual disk. 
     *
     * @param sourceGuestName The name of the guest that owns the disk to be shared.
     * @param sourceVaddr The address of the disk to be shared.
     * @param destGuestName The guest that will obtain the link to the shared disk.
     * @param destVaddr The virtual address that will be created and linked to the shared disk.
     * @param accMode The mode that the disk will be shared in.
     * @param passwordNeededToLink Optional.  Password used to access the disk.  Pass empty string to ignore this password.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskShare(String sourceGuestName, String sourceVaddr, String destGuestName, String destVaddr, String accMode, String passwordNeededToLink)
        throws SMAPIException
    {
	String parm04 = destGuestName;
	String parm05 = "1";
	String parm06 = destVaddr;
	String parm07 = sourceGuestName;
	String parm08 = sourceVaddr;
	String parm09 = accMode;
	String parm10 = passwordNeededToLink;

	LinkedList tokens = connectRequest(IMAGE_DISK_SH,   parm04 + " " + parm05 + " " + parm06
						    + " " + parm07 + " " + parm08 + " " + parm09
						    + " " + parm10);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Share",  rc,  rs);
    }
   
    /**
     * Corresponds to the SMAPI procedure IMAGE_DISK_SHARE.  Creates a link to
     * a guests virtual disk.  This call modfies the user directory entry for this
     * guest.
     *
     * @param sourceGuestName The name of the guest that owns the disk to be shared.
     * @param sourceVaddr The address of the disk to be shared.
     * @param destGuestName The guest that will obtain the link to the shared disk.
     * @param destVaddr The virtual address that will be created and linked to the shared disk.
     * @param accMode The mode that the disk will be shared in.
     * @param passwordNeededToLink Optional.  Password used to access the disk.  Pass empty string to ignore this password.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void imageDiskShareDM(String sourceGuestName, String sourceVaddr, String destGuestName, String destVaddr, String accMode, String passwordNeededToLink)
        throws SMAPIException
    {
	String parm04 = destGuestName;
	String parm05 = "2";
	String parm06 = destVaddr;
	String parm07 = sourceGuestName;
	String parm08 = sourceVaddr;
	String parm09 = accMode;
	String parm10 = passwordNeededToLink;

	LinkedList tokens = connectRequest(IMAGE_DISK_SH_DM,   parm04 + " " + parm05 + " " + parm06
						       + " " + parm07 + " " + parm08 + " " + parm09
						       + " " + parm10);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Share DM",  rc,  rs);
    }

    /**
     * Corresponds to the SMAPI procedure IMAGE_DISK_UNSHARE.  Removes a
     * link to another guests disk.
     *
     * @param guestUsingDisk Name of the guest linking to the disk.
     * @param vaddrOfGuestUsingDisk Virtual address of disk with respect to the guest with the link.
     * @param guestOwningDisk Guest that owns the disk being linked to
     * @param vaddrOfGuestOwningDisk Virtual address of disk with respect to the guest that owns the disk.
     */
    public void imageDiskUnshare(String guestUsingDisk, String vaddrOfGuestUsingDisk, String guestOwningDisk, String vaddrOfGuestOwningDisk)
        throws SMAPIException
    {
	String parm04 = guestUsingDisk;
	String parm05 = "1";
	String parm06 = vaddrOfGuestUsingDisk;
	String parm07 = guestOwningDisk;
	String parm08 = vaddrOfGuestOwningDisk;

	LinkedList tokens = connectRequest(IMAGE_DISK_UNSH,   parm04 + " " + parm05 + " " + parm06
						    + " " + parm07 + " " + parm08);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Unshare",  rc,  rs);
    }
   
    /**
     * Corresponds to the SMAPI procedure IMAGE_DISK_UNSHARE.  Removes a
     * link to another guests disk.  This call modfies the user directory entry 
     * for this guest.
     * @param guestUsingDisk Name of the guest linking to the disk.
     * @param vaddrOfGuestUsingDisk Virtual address of disk with respect to the guest with the link.
     * @param guestOwningDisk Guest that owns the disk being linked to
     * @param vaddrOfGuestOwningDisk Virtual address of disk with respect to the guest that owns the disk.
     */
    public void imageDiskUnshareDM(String guestUsingDisk, String vaddrOfGuestUsingDisk, String guestOwningDisk, String vaddrOfGuestOwningDisk)
        throws SMAPIException
    {
	String parm04 = guestUsingDisk;
	String parm05 = "2";
	String parm06 = vaddrOfGuestUsingDisk;
	String parm07 = guestOwningDisk;
	String parm08 = vaddrOfGuestOwningDisk;

	LinkedList tokens = connectRequest(IMAGE_DISK_UNSH_DM,   parm04 + " " + parm05 + " " + parm06
									+ " " + parm07 + " " + parm08);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Image Disk Unshare DM",  rc,  rs);
    } 

    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_LAN_CONNECT.  
     * Creates a virtual LAN (if it does not already exist) and connects a 
     * VNIC to it.  The VNIC is also created on the fly.  This is
     * the only official SMAPI method to couple a VNIC to a VLAN.  Many parameters
     * to the underlying SMAPI call are set to sane defaults.  The resulting LAN
     * will always be a QDIO, IP based SYSTEM owned LAN.  The resulting VNIC will
     * have allocated 3 devices.
     * @param lanName The name of the LAN to couple to.
     * @param guestName Guest owning the NIC.
     * @return int 0 guest added to pre-existing guest LAN, 1 object directory offline, 2 new lan created
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualLANCreateAndConnect(String lanName, String guestName)
        throws SMAPIException
    {
	String vaddr = getLowestFreeAddress(guestName, "3");
	LinkedList tokens = connectRequest(VIRT_NET_LAN_CON, guestName + " " + "3" + " " + vaddr + " " + lanName + " " + "SYSTEM" + " " + "QDIO" + " " + "3" + " " + "IP");
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	 
	if (rc == 0 && rs == 0)  return 0; // guest added to pre-existing guest LAN
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 20) return 2; // new guest LAN created and guest added
	else errorCheck("LAN Create and connect",  rc,  rs);

	return -1; //Never get here
    }
    
    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_LAN_CONNECT_DM.
     * This call updates the guest directory entry.  
     * Creates a virtual LAN (if it does not already exist) and connects a 
     * VNIC to it.  The VNIC is also created on the fly.  This is
     * the only official SMAPI method to couple a VNIC to a VLAN.  Many parameters
     * to the underlying SMAPI call are set to sane defaults.  The resulting LAN
     * will always be a QDIO, IP based SYSTEM owned LAN.  The resulting VNIC will
     * have allocated 3 devices.
     * @param lanName The name of the LAN to couple to.
     * @param guestName Guest owning the NIC.
     * @param vaddr guestName virtual address of the new virtual NIC.
     * @return int 0 guest added to pre-existing guest LAN, 1 object directory offline, 2 new lan created
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualLANCreateAndConnectDM(String lanName, String guestName, String vaddr)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(VIRT_NET_LAN_CON_DM, guestName + " " + "3" + " " + vaddr + " " + lanName + " " + "SYSTEM" + " " + "QDIO" + " " + "3" + " " + "IP");
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	 
	if (rc == 0 && rs == 0) return 0; // guest added to pre-existing guest LAN
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 20) return 2; // new guest LAN created and guest added
	else errorCheck("LAN Create and conect DM",  rc,  rs);

	return -1;  //Never get here
    }
    
    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_LAN_DISCONNECT.  
     * Disconnects a virtual NIC from a guest LAN and deletes the LAN if there 
     * are no more NICs connected to it.
     * @param lanName The LAN to delete the NIC from.
     * @param guestName Guest owning the NIC to delete.
     * @param vaddr Base address of the NIC to delete.
     * @return int 0 guest deleted from lan, 1 object directory is offline, 2 lan deleted
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualLANDisconnectAndDelete(String lanName, String guestName, String vaddr)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(VIRT_NET_LAN_DS, guestName + " " + "3" + " " + vaddr + " " + lanName + " " + "SYSTEM" + " " + "QDIO");
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	 
	if (rc == 0 && rs == 0) return 0; // guest deleted from LAN
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 24) return 2; // guest LAN deleted
	else errorCheck("LAN Disconnect and delete",  rc,  rs);
	
	return -1; //Never get here
    }
  
    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_LAN_DISCONNECT_DM.  
     * Disconnects a virtual NIC from a guest LAN and deletes the LAN if there 
     * are no more NICs connected to it.  This call modifies the guests drectory
     * entry.
     * @param lanName The LAN to delete the NIC from.
     * @param guestName Guest owning the NIC to delete.
     * @param vaddr Base address of the NIC to delete.
     * @return int 0 guest deleted from lan, 1 object directory is offline, 2 lan deleted
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualLANDisconnectAndDeleteDM(String lanName, String guestName, String vaddr)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(VIRT_NET_LAN_DS_DM, guestName + " " + "3" + " " + vaddr + " " + lanName + " " + "SYSTEM" + " " + "QDIO");
        
	//get return and reason codes
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	 
	if (rc == 0 && rs == 0) return 0; // guest deleted from LAN
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 24) return 2; // guest LAN deleted
	else errorCheck("LAN Disconnect and delete DM",  rc,  rs);
	
	return -1; //Never get here
    }
    
      /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_VSWITCH_CONNECT.  
     * Creates a virtual switch (if it does not already exist) and connects a 
     * VNIC to it.  The VNIC is also created on the fly.  This is
     * the only official SMAPI method to couple a VNIC to a VSWITCH.
     * @param switchName name of the VSWITCH to create/connect NIC to.
     * @param guestName Guest to own NIC.
     * @param vaddr Base address for NIC.
     * @param portname The port name to connect the real device to
     * @param raddr The physical address of the real OSA device
     * @return int 0 guest connected to switch, 1 object directory is offline, 2 switch created
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualSwitchCreateAndConnect(String switchName, String guestName, String vaddr, String portname, String raddr)
        throws SMAPIException
    {
	if (portname == null || portname.equals("")) portname = "&";
	if (raddr == null || raddr.equals("")) raddr = "NONE";
	
	//We start at 4 because SMAPI sees the first parameter listed below as the 4th.
	//Dont ask :)  If you get an RC=24 and an RS=xxyy, xx is param number.  If it says
	//4, you REALLY want to check the FIRST parameter.
	String parm04 = guestName;	// Image
	String parm05 = "3";		// 3=synamic
	String parm06 = vaddr;		// Virt addr
	String parm07 =	switchName;	//Switch name
	String parm08 = portname; 	//port
	String parm09 =	raddr;		//real
	String parm10 =	"ACTIVATE";	//connection val
	String parm11 =	"&";		//queue stor lim
	String parm12 =	"&";		//controller
	String parm13 =	"3";		//# devices
	String parm14 =	"&";		//chpid
	String parm15 =	"&";		//mac id
	String parm16 =	"PRIROUTER";	//route val
	String parm17 =	"IP";		//transport type
	
	LinkedList tokens = connectRequest(VIRT_NET_VSW_CON, parm04 + " " + parm05 + " " + parm06 + " " +
						parm07 + " " + parm08 + " " + parm09 + " " + parm10 + " " +
						parm11 + " " + parm12 + " " + parm13 + " " + parm14 + " " +
						parm15 + " " + parm16 + " " + parm17);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if (rc == 0 && rs == 0) return 0; //guest Connected to VSwitch
	else if (rc == 0 && rs == 8) return 1; //object directory is offline
	else if (rc == 0 && rs == 40) return 2; //new VSwitch created    
	else errorCheck("VSwitch Create and connect",  rc,  rs);
	
	return -1; //never get here
    }
    
    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_VSWITCH_CONNECT_DM.  
     * This call modifies the guests directory entry.  This call connects a guest to a VSwitch
     *
     * @param switchName name of the VSWITCH to create/connect NIC to.
     * @param guestName Guest to own NIC.
     * @param vaddr Base address for NIC.
     * @param portname The port name to connect the real device to
     * @param raddr The physical address of the real OSA device
     * @return int 0 guest added to switch, 1 object directory is offline, 2 switch created
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualSwitchCreateAndConnectDM(String switchName, String guestName, String vaddr, String portname, String raddr)
        throws SMAPIException
    {
	if (portname == null || portname.equals("")) portname = "&";
	if (raddr == null || raddr.equals("")) raddr = "NONE";
	
	//We start at 4 because SMAPI sees the first parameter listed below as the 4th.
	//Dont ask :)  If you get an RC=24 and an RS=xxyy, xx is param number.  If it says
	//4, you REALLY want to check the FIRST parameter.
	String parm04 = guestName;	// Image
	String parm05 = "3";		// 3=dynamic
	String parm06 = vaddr;		// Virt addr
	String parm07 =	switchName;	//Switch name
	String parm08 = portname; 	//port
	String parm09 =	raddr;		//real
	String parm10 =	"ACTIVATE";	//connection val
	String parm11 =	"&";		//queue stor lim
	String parm12 =	"&";		//controller
	String parm13 =	"3";		//# devices
	String parm14 =	"&";		//chpid
	String parm15 =	"&";		//mac id
	String parm16 =	"PRIROUTER";	//route val
	String parm17 =	"IP";		//transport type
	
	LinkedList tokens = connectRequest(VIRT_NET_VSW_CON_DM, parm04 + " " + parm05 + " " + parm06 + " " +
						parm07 + " " + parm08 + " " + parm09 + " " + parm10 + " " +
						parm11 + " " + parm12 + " " + parm13 + " " + parm14 + " " +
						parm15 + " " + parm16 + " " + parm17);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	if (rc == 0 && rs == 0) return 0; //guest Connected to VSwitch
	else if (rc == 0 && rs == 8) return 1; //object directory is offline
	else if (rc == 0 && rs == 40) return 2; //new VSwitch created    
	else errorCheck("VSwitch Create and connect DM",  rc,  rs);
	
	return -1; //never get here
    }
    
    /**
     * UNOFFICIAL: Creates a VSwitch with the given parameters.  Does not require you to 
     * create or couple any virtual NIC at the time of creation, unlike the
     * official SMAPI call.
     * @param switchName Name of the VSwitch you are creating
     * @param portname The portname for the real device attached to this VSwitch.
     * @param raddr The address for the real device attached to this VSwitch.
     */
    public void virtualSwitchCreate(String switchName, String portname, String raddr)
        throws SMAPIException
    {
	if (portname == null || portname.equals("")) portname = "&";
	if (raddr == null || raddr.equals("")) raddr = "NONE";
	
	//We start at 4 because SMAPI sees the first parameter listed below as the 4th.
	//Dont ask :)  If you get an RC=24 and an RS=xxyy, xx is param number.  If it says
	//4, you REALLY want to check the FIRST parameter.
	String parm04 = switchName;
	String parm05 = portname; 	//port
	String parm06 = raddr;		//real
	
	LinkedList tokens = connectRequest(VIRT_SWIT_CREATE, parm04 + " " + parm05 + " " + parm06);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	if (rc == 99) throw new VSwitchAlreadyExistsException("VSwitch already exists.");
	errorCheck("VSwitch Create",  rc,  rs);
    }
    
    /**
     * UNOFFICIAL: Creates a guest LAN.  Does not require you to 
     * create or couple any virtual NIC at the time of creation, unlike the
     * official SMAPI call.
     * @param lanName Name of the guest LAN you are creating
     */
    public void virtualLANCreate(String lanName)
        throws SMAPIException
    {
	//We start at 4 because SMAPI sees the first parameter listed below as the 4th.
	//Dont ask :)  If you get an RC=24 and an RS=xxyy, xx is param number.  If it says
	//4, you REALLY want to check the FIRST parameter.
	String parm04 = lanName;

	LinkedList tokens = connectRequest("SSSLMAKE", parm04);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	if (rc == 99) throw new LANAlreadyExistsException("LAN Create: Lan already exists.");
	errorCheck("LAN Create",  rc,  rs);
    }
   
    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_VSWITCH_SET.  The underlying 
     * SMAPI call allows the caller to Set or change many different VSWITCH 
     * parameters.  Our implementation only allows the use of VSwitch Set to 
     * grant/revoke guest access to the given VSwitch.
     * @param switchName name of the VSWITCH to modify.
     * @param grantID userID to grant access to the VSWITCH.
     * @param revokeID userID to revoke access to the VSWITCH.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void virtualSwitchSet(String switchName, String grantID, String revokeID)
        throws SMAPIException
    {
	if(grantID == null || grantID.equals("")) grantID = "&";
	if(revokeID == null || revokeID.equals("")) revokeID = "&";

	//We start at 4 because SMAPI sees the first parameter listed below as the 4th.
	//Dont ask :)  If you get an RC=24 and an RS=xxyy, xx is param number.  If it says
	//4, you REALLY want to check the FIRST parameter.
	String parm04 = "SYSTEM";		// Image
	String parm05 = switchName;	// Switch name
	String parm06 = grantID;	// Grant ID
	String parm07 =	"&";	// Vlan ID
	String parm08 = revokeID; 	//Revoke ID
	String parm09 = "&";		//port
	String parm10 =	"&";		//real
	String parm11 =	"&";		//connection val
	String parm12 =	"&";		//queue stor lim
	String parm13 =	"&";		//controller
	String parm14 = "&";		//route val
	
	LinkedList tokens = connectRequest(VIRT_NET_VSW_SET, parm04 + " " + parm05 + " " + parm06 + " " +
						parm07 + " " + parm08 + " " + parm09 + " " + parm10 + " " +
						parm11 + " " + parm12 + " " + parm13 + " " + parm14);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	    
	errorCheck("VSwitch set",  rc,  rs);

    }

    /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_VSWITCH_DISCONNECT.  
     * Disconnects a virtual NIC from a VSWITCH and deletes the VSWITCH if there 
     * are no more NICs connected to it.
     *
     * @param guestName Guest owning the NIC to delete.
     * @param switchName Name of VSWITCH.
     * @param vaddr Base address of NIC.
     * @return int 0 for guest removed from VSwitch, 1 object directory is offline, 2 VSwitch deleted
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualSwitchDisconnectAndDelete(String guestName, String switchName, String vaddr)
        throws SMAPIException
    {
 	LinkedList tokens = connectRequest(VIRT_NET_VSW_DS, guestName + " " + "3" + " " + vaddr + " " + switchName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if (rc == 0 && rs == 0) return 0; // guest removed from VSwitch
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 44) return 2; //  VSwitch deleted
	else errorCheck("VSwitch disconnect and delete",  rc,  rs);

	return -1; //Never get here
    }

   /**
     * Corresponds to the SMAPI procedure VIRTUAL_NETWORK_VSWITCH_DISCONNECT.  
     * Disconnects a virtual NIC from a VSWITCH.  This call modifies the guests
     * directory entry.
     *
     * @param guestName Guest owning the NIC to delete.
     * @param switchName Name of VSWITCH.
     * @param vaddr Basee address of NIC.
     * @return int 0 guest removed from VSwitch, 1 object directory offline, 2 VSwitch deleted 
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public int virtualSwitchDisconnectAndDeleteDM(String guestName, String switchName, String vaddr)
        throws SMAPIException
    {
 	LinkedList tokens = connectRequest(VIRT_NET_VSW_DS_DM, guestName + " " + "3" + " " + vaddr + " " + switchName);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if (rc == 0 && rs == 0) return 0; // guest removed from VSwitch
	else if (rc == 0 && rs == 8) return 1; // object directory is offline
	else if (rc == 0 && rs == 44) return 2; //  VSwitch deleted
	else errorCheck("VSwitch disconnect and delete DM",  rc,  rs);

	return -1; //Never get here
    }
     
    /**
     * UNOFFICIAL:  Given a guest to use as a model, find an appropriate page
     * range for a new shared segment of the given size.  The returned range will
     * be above the models address space to ensure that no conflicts occur.
     * @param segmentSizeInMB Size of desired shared segment.
     * @param guestName Guest to use as model.
     * @return String a string in the format nnnn-mmmm where nnnn and mmmm are both hexadecimal
     * page numbers.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public String sharedStorageFindPageRange(String segmentSizeInMB, String guestName)
	throws SMAPIException
    {
	LinkedList tokens = connectRequest(FIND_STORAGE, "0" + " " + segmentSizeInMB + " " + "false" + " " + guestName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if(rc == 600 && rs == 128) throw new SMAPIException("Finding shared storage gap: no free gaps found that match the given parameters!");
	errorCheck("Finding shared storage gap",rc,rs);
		
	return (String)tokens.getFirst() ;
    }
    
    /*
     * Corresponds to the SMAPI procedure SHARED_STORAGE_CREATE.
     * Creates a shared storage segment.  All segements created are restricted.
     *
     * @param guestName Guest name owning the segment.
     * @param segmentName Segment name.
     * @param pageRange Page range.
     * @param accessDescriptor Page access descriptor.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void sharedStorageCreate(String guestName, String segmentName, String pageRange, String accessDescriptor)
        throws SMAPIException
    {
	String parm04 = guestName;
	String parm05 = segmentName;
	String parm06 = pageRange;
	String parm07 = accessDescriptor;
	String parm08 = "RSTD";  //ALWAYS RESTRICTED.
	String parm09 = "&";
	
	LinkedList tokens = connectRequest(SS_CREATE,	parm04 + " " + parm05 + " " + parm06 + " " +
							parm07 + " " + parm08 + " " + parm09);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if(rc == 600 && rs == 128) throw new SMAPIException("Finding shared storage gap: no free gaps found that match the given parameters!");
	errorCheck("Creating shared storage segment",rc,rs);
    }

    /**
     * Corresponds to the SMAPI procedure SHARED_STORAGE_DELETE.
     * Deletes a shared storage segment.
     *
     * @param guestName Guest owning the segment.
     * @param segmentName name of segment.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void sharedStorageDelete(String guestName, String segmentName)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(SS_DELETE, guestName + " " + segmentName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Deleting shared storage segment",rc,rs);        
    }

    /** UNOFFICIAL:  We currently use an unofficial call to do this.
     * Corresponds to the SMAPI procedure SHARED_STORAGE_QUERY.
     * Querries and returns info for all shared segments on the system.
     *
     * @return LinkedList corresponding to a Q NSS ALL command.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList sharedStorageQuery()
        throws SMAPIException
    {
        //  '*' will query ALL NSS's
	//LinkedList tokens = connectRequest(SS_QUERY, "VSMSTEVE *");
	
	//TODO: Use official version.
	//Not using official version because code is already written to parse
	//this one more easily.
	LinkedList tokens = connectRequest("SSSQNSS",  ""); 
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Querying shared storage segments",rc,rs);   
	
	return tokens;
    }
    
    /**
     * Corresponds to the SMAPI procedure SHARED_STORAGE_ACCESS_ADD.
     * Adds restricted access to a shared storage segment.
     *
     * @param guestName guest to give access.
     * @param segmentName Segment name.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void sharedStorageAddAccess(String guestName, String segmentName)
        throws SMAPIException
    {
       LinkedList tokens = connectRequest(SS_ACC_ADD_DM, guestName + " " + segmentName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Adding access for shared storage segment",rc,rs);      
    }

    /**
     * Corresponds to the SMAPI procedure SHARED_STORAGE_ACCESS_REMOVE.
     * Removes access to a shared storage segment.
     *
     * @param guestName Guest for whom to remove access.
     * @param segmentName Segment name.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void sharedStorageRemoveAccess(String guestName, String segmentName)
        throws SMAPIException
    {
        LinkedList tokens = connectRequest(SS_ACC_REM_DM, guestName + " " + segmentName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if( !(rc == 424 && rs == 8) )
	errorCheck("Removing access for shared storage segment",rc,rs); 
    }
    
    /**
     * Queries access information for a shared storage segment.
     *
     * @param guestToQuery Name of the guest for whom to query shared storage segments
     * @param segToQuery Name of the segment to query information for
     * @return boolean Whether the guest is on the access list for the shared storage segment
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public boolean sharedStorageQueryAccess(String guestToQuery, String segToQuery)
        throws SMAPIException
    {
	LinkedList tokens = connectRequest(SS_ACC_QUE_DM, guestToQuery + " " + segToQuery);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	if(rc == 0 && rs == 20) return false; //Not authorized.
	if(rc == 0 && rs == 0) return true; //Authorized
	
	errorCheck("Querying access for shared storage segment",rc,rs); 
	
	return false; //Should never get here!
    } 
    
    /**
     * UNOFFICIAL: Returns the results of QUERY VIRTUAL for the given guest.
     * Can be parsed to get the used and free virtual addresses.
     * 
     * The return value has the following format:
     *  USERNAME
     *  0009 ON LDEV L0005   TERM START HOST TCPIP    FROM 9.60.66.183 
     *  0009 CL T NOCONT NOHOLD COPY 001    READY FORM STANDARD        
     *  0009 TO OP1      RDR DIST OPERATOR  FLASHC 000 DEST OFF        
     *  0009 FLASH       CHAR       MDFY       0 FCB       LPP OFF     
     *  0009 3215   NOEOF OPEN 0046 NOKEEP NOMSG NONAME                
     *  0009 SUBCHANNEL = 0011 
     *  DASD
     *  0190 3390 510RES R/O        107 CYL ON DASD  0700 SUBCHANNEL = 0012     
     *  NICS
     *  1030 ON NIC  1030  UNIT 000 SUBCHANNEL = 0000                        
     *  1030 QDIO-ELIGIBLE       QIOASSIST NOT AVAILABLE                                    
     *
     * @param guestName Guest for which to query virtual adddresses.
     * @return LinkedList of strings, where each string is one line of results from Query Virtual
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList returnQueryVirtual (String guestName)
	throws SMAPIException
    {
	LinkedList tokens = connectRequest(QUERY_VIRTUAL, guestName);
	
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Query Virtual",  rc,  rs);
	
	return tokens;
    }
    
    /**
     * UNOFFICIAL: Returns the lowest free virtual address for a particular user.                   
     * @param guestName Guest to query.
     * @param size Size of free virtual address space needed (i.e. NICS need 3)
     * @return String Four digit hexadecimal number representing the starting address of the free virtual address space.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public String getLowestFreeAddress (String guestName, String size)
	throws SMAPIException
    {
        LinkedList tokens = connectRequest(LOW_FREE_ADDR, guestName + " " + size);

	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

 	if (rc != 0) errorCheck("Lowest Address Query",  rc,  rs);
	
	return (String)tokens.removeFirst();
    }
    
    /**
     * UNOFFICIAL: This function returns the members of a specified VSwitch, or all
     * VSwitches on the system.  The members list is obtained by issuing a Q 
     * VSWITCH netName ACC and reading the access lsit.  The results are returned
     * in a LinkedList of {@link live.dto.Network}s.
     *
     * @param netName Name of the network to query, use "" or "&" to query all networks
     * @return LinkedList of {@link live.dto.Network}s, each with containing only network name and members
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList queryNetworkMembers(String netName)
	throws SMAPIException
    {
	if (netName == null || netName.equals("")) netName = "&";
        LinkedList tokens = connectRequest(QUER_NET_MEMBERS, netName);
 
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
	errorCheck("Network Members Query",  rc,  rs);
	
	LinkedList ret = new LinkedList();
	String token = (String)tokens.removeFirst();
	
	//We expect the first token to be a delimiter ": "
	if(!token.endsWith(": ") )
	{
	    throw new SMAPIException("Query network members: Should have seena ':' character!");
	}
	
	//Iterate through the returned list of strings
	ListIterator iter = tokens.listIterator();
	
	while(iter.hasNext())  
	{
		token = (String)iter.next();
		Network n = new Network(token, 1); 
		
		while(! token.equals(": ") && iter.hasNext())
		{
		    token = (String)iter.next();
		    //Don't list system as a user, since we can't perform any network actions on it
		    if (!token.equals("SYSTEM"))  n.addMember(token);
		}
		
		ret.add(n);
	}
		
	return ret;
    } 
    
    /**
     * UNOFFICIAL: This function returns the virtual address of a specified NIC 
     * for a specified user on a specified network .
     *
     * @param netName Name of the network the NIC is attached to.
     * @param guestName Name of the guest to query on the given network
     * @return String 4 digit hexadecimal number that is the starting address of the NICs address space.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public String getNICaddress(String netName, String guestName)
	throws SMAPIException
   {
       LinkedList tokens = connectRequest(QUER_NIC_ADDR, netName + " " + guestName);
       
       int rc = Integer.parseInt((String)tokens.removeFirst());
       int rs = Integer.parseInt((String)tokens.removeFirst());
       
	errorCheck("NIC Address Query",  rc,  rs);
       
       return (String)tokens.removeFirst();
   }
   
    /**
     * UNOFFICIAL: This function returns all information from the SMAPI namelist
     * file which is used to store the SMAPIs namelists and their members.
     * @return LinkedList of {@link live.dto.NameList}s
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public LinkedList nameListQueryAll()
	throws SMAPIException
    {
		LinkedList tokens = connectRequest(NAME_LIST_Q_ALL, null);
		
		int rc=Integer.parseInt((String)tokens.removeFirst());
		int rs=Integer.parseInt((String)tokens.removeFirst());
			
		errorCheck("NameList Query all",  rc,  rs);
		
		LinkedList ret = new LinkedList();
		
		if(! ":".equals((String)tokens.removeFirst()) )
		{
			throw new SMAPIException("NameList Query all: Should have seen a ':'!");
		}
		
		while(tokens.size() > 1)  // >1 so we ignore the last element
		{
			String token = (String)tokens.removeFirst();

			NameList list = new NameList(token);
			
			// the following line cannot cause removeFirst to complain that it's empty because
			// when we entered the loop we had at least 2 elements in the list, and this is only
			// the second we're removing
			token = (String)tokens.removeFirst();
			while(! token.equals(":") && tokens.size() > 0)
			{
				list.addElement(token);
				token = (String)tokens.removeFirst();
			}
			
			ret.add(list);
		}
		
		return ret;
    }
	
    /**
    * UNOFFICIAL: This function returns a LinkedList of Strings.  Each String 
    * represents a guest who is currently active.
    * @return LinkedList of Strings (guest names)
    * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
    */
    public LinkedList activeImagesQuery()
	throws SMAPIException
    {
		LinkedList tokens = connectRequest(ACTIVE_IMAGE_QUERY, null);
		
		int rc=Integer.parseInt((String)tokens.removeFirst());
		int rs=Integer.parseInt((String)tokens.removeFirst());
			
		errorCheck("Active Images query",  rc,  rs);
		
		return tokens;
    }
       
    /**
     * UNOFFICIAL: Deletes a VSwitch from the system.  Unlike the official SMAPI
     * call, this one does not require that you decouple a specific virtual NIC
     * before deleting the VSwitch.
     * @param switchName Name of the switch to delete.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void virtualSwitchDelete(String switchName)
	throws SMAPIException
    {
       LinkedList tokens = connectRequest(VSW_DEL, switchName);

	//get return and reason codes
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());

	//Is there an error?
	errorCheck("VSwitch Delete",  rc,  rs);

	//Extra error checking
	if (rc == 50 && rs == 0) throw new SMAPIException("VSwitch delete!: could not delete VSwitch.  Unknown reason.");

	//Extra error checking
	if (rc == 50 && rs == 0) throw new SMAPIException("VSwitch detatch!: could not detach NIC.  Unknown reason.");
    }
   
    /**
     * UNOFFICIAL: Detatches a guest's virtual NIC from the VSwitch it is attched 
     * to.  Unlike the official SMAPI call, this one does not delete the VSwitch 
     * if no online user in connected to it after the detatch.
     * @param guestName Name of guest obtaining the NIC to detatch.
     * @param vaddr The virtual address of the NIC to detatch.
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    public void virtualSwitchDetatch(String guestName, String vaddr)
	throws SMAPIException
    {
	LinkedList tokens = connectRequest(VSW_DET, guestName + " " + vaddr);
        
	int rc=Integer.parseInt((String)tokens.removeFirst());
	int rs=Integer.parseInt((String)tokens.removeFirst());
	
 	errorCheck("VSwitch Detatch",  rc,  rs);
    }
      
      
   /************ PRIVATE ***************/

   /** Executes the requested SMAPI function given its name and parameters.  Will
    * return a linked list of strings.  The first sting will be the RC, the second
    * string will be the RS.  The number of and meaning of the rest of the strings 
    * in the list is dependant on the exact SMAPI call being made.  The parms String
    * should be space delimited.  If you wish to ignore an optional parameter a '&'
    * character should appear in its place in the parms String.
    * @param functionName The short name of the SMAPI function to be called.
    * @param parms The space delimited list of paramters for the given SMAPI call.
    * @return LinkedList RC, RS, and return information from the call
    * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
    */
    private LinkedList connectRequest(String functionName, String parms)
	throws SMAPIException
    {
	Socket SMAPISocket = null;
	DataOutputStream out = null;
	BufferedReader in = null;
	//LinkedList tokens = new LinkedList();
	String rawData = null;
	
	try 
	{
	    SMAPISocket = new Socket(hostName, port);
	    out = new DataOutputStream(SMAPISocket.getOutputStream());
	    
	    // The ISO-8859-1 codepage specification seems to be neede inorder for
	    // this to work correctly on zLinux.  No further explanation is 
	    // available.
	    in = new BufferedReader(new InputStreamReader(SMAPISocket.getInputStream(), "ISO-8859-1"));
	    
	    //send data to SMAPI server
	    out.write(functionName.getBytes());
	    out.write(B_NULL);
	    out.write(userName.getBytes());
	    out.write(B_NULL);
	    out.write(password.getBytes());
	    out.write(B_NULL);

	    // We only want to write parameters if there are parameters.
	    if (parms != null)
	    {
		StringTokenizer st = new StringTokenizer(parms);
		while (st.hasMoreTokens())
		{
		    String s=st.nextToken();
		    if (!s.equals("&"))
		    {
			out.write(s.getBytes());
		    }
		    else
		    {
			out.write(0x20);
		    }
		    out.write(B_NULL);
		}
	    }
	    
	    //Write end of message string which consists of 0x01 0x01
	    out.write(B_LINETERM); // First  0x01
	    out.write(B_LINETERM); // Second 0x01
	    
	    //receive return string from SMAPI server
	    rawData = in.readLine();
	    while(in.ready())
	    {
		rawData += in.readLine();
	    }
	    
	    //close connection
	    out.close();
	    in.close();
	    SMAPISocket.close();
	}  
	catch (UnknownHostException e) 
	{
            throw new SMAPIException(e.getMessage());
        } 
	catch (IOException e) 
	{
	    throw new SMAPIException(e.getMessage());            
        }
	
	//return information from SMAPI server
	if (rawData == null) throw new SMAPIException("Error reading data from SMAPI server. rawData is null!");      
	return parseServerStream(rawData);
    }
    
    /** Executes the requested SMAPI function given its name and parameters.  Will
    * return a linked list of strings.  The first sting will be the RC, the second
    * string will be the RS.  The number of and meaning of the rest of the strings 
    * in the list is dependant on the exact SMAPI call being made.  The parms String
    * is assumed to already be null delimited and correctly formatted to be sent
    * off to the server.  SMAPI calls with special formatting needs should use this
    * call.  The parms String is passed to the SMAPI server "AS IS".
    * @param functionName The short name of the SMAPI function to be called.
    * @param parms The paramters for the given SMAPI call, passed "AS IS".
    * @return LinkedList RC RS and return information from the call
    * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
    */
    private LinkedList connectRequestRAW(String functionName, String parms)
	throws SMAPIException
    {
	Socket SMAPISocket = null;
	DataOutputStream out = null;
	BufferedReader in = null;
	//LinkedList tokens = new LinkedList();
	String rawData = null;
	
	try 
	{
	    SMAPISocket = new Socket(hostName, port);
	    out = new DataOutputStream(SMAPISocket.getOutputStream());
	    in = new BufferedReader(new InputStreamReader(SMAPISocket.getInputStream()));
	    
	    //send data to SMAPI server
	    out.write(functionName.getBytes());
	    out.write(B_NULL);
	    out.write(userName.getBytes());
	    out.write(B_NULL);
	    out.write(password.getBytes());
	    out.write(B_NULL);
	    
	    // We only want to write parameters if there are parameters.
	    if (parms != null)
	    {
		out.write(parms.getBytes());
	    }
	    
	    //Write end of message string
	    out.write(B_LINETERM);
	    out.write(B_LINETERM);
	    
	    //receive return string from SMAPI server
	    rawData = in.readLine();
	    while(in.ready())
	    {
		rawData += in.readLine();
	    }
	    
	    //close the connection
	    out.close();
	    in.close();
	    SMAPISocket.close();
	}  
	
	catch (UnknownHostException e) 
	{
            throw new SMAPIException(e.getMessage());
        } 
	catch (IOException e) 
	{
	    throw new SMAPIException(e.getMessage());            
        }
	
	//return information from SMAPI server
	if (rawData == null) throw new SMAPIException("Error reading data from SMAPI server. rawData is null!");      
	return parseServerStream(rawData);
    }
    
    /**
     * Parses raw data returned from the SMAPI server and returns a LinkedList
     * of Strings representing the returned data.
     * @param data Raw data from the SMAPI server.
     * @return LinkedList of Strings representing the returned data.
     */
    private LinkedList parseServerStream(String data)
    {
	/* Data format:
	 
	 rc    space rs   space
	 NULL  NULL  NULL JUNK
	 [real data]
	 NULL  NULL  NULL JUNK
	 [real data]
	 ...
	
	 update: the NULL NULL NULL JUNK entries are actually a 4-byte size that
	 tells you how big the next element/value/whatever is.  If you choose to
	 do something with this be careful because the 4-byte chunk underwent
	 EBCDIC to ASCII translation just like everything else.  You'll need to
	 translate it back to EBCDIC before you interpret it as a binary number
	 representing the size of the data following it.
	 */
	
	LinkedList returnTokens = new LinkedList();
	StringBuffer sbData = new StringBuffer(data);
	
	// Read rc and rs 
	String rc = readUntilSeeChar(sbData, B_SPACE);
	sbData.deleteCharAt(0); //Eat the space
	
	String rs = readUntilSeeChar(sbData, B_SPACE);
	sbData.deleteCharAt(0); //Eat the space
	
	returnTokens.add(rc);
	returnTokens.add(rs);
	
	// Read args!
    	while(sbData.length() > 4)
	{
	    //Eat four chars
	    sbData.delete(0, 4);
	    String curArg = readUntilSeeChar(sbData, B_NULL);

	    returnTokens.add(curArg);
	}
	
	return returnTokens;
    }
    
    /**
     * Read from a string buffer until we need a given character, remove and
     * return all the read data.
     * @param sb A StringBuffer containing the data to search through.
     * @param character The character we are looking for.
     * @return String the data that was found before the requested character.
     */
    private String readUntilSeeChar(StringBuffer sb, byte character)
    {
	byte[] b = {character};
	int nullIndex = sb.indexOf( new String(b));
	
	//If we dont find the character, delete/return the whole string.
	if(nullIndex == -1) 
	{
	    String retVal = sb.toString();
	    sb.setLength(0);
	    return retVal;
	}
	
	//If we do find the character, only delete/return the portion before it.
	String subString = sb.substring(0, nullIndex);
	sb.delete(0, nullIndex);
	
	return subString;
    }   
    
    /**
     * Given an RC/RS pair, compare them against known errors and throw an
     * appropriate exception if an error is detected.  The message for the
     * exception will contain a desc String passed in.  The desc Strng is meant to 
     * contain a description of the function calling errorCheck() so the programmer
     * will have a little more information when debugging.  With proper descriptive
     * messages, the error mesasges given from this function could possible be 
     * shown directly to and end-user.  If this function returns without throwing 
     * an exception, no error was detected.
     * @param desc String describing the function calling errorCheck.
     * @param rc SMAPI return code
     * @param rs SMAPI result code
     * @throws SMAPIException An error occurred.  This is the parent exception for all other exceptions in this library. 
     */
    private void errorCheck(String desc, int rc, int rs)
	throws SMAPIException
    {
	desc += ": ";
	
	//Make sure there was an error.  If not, return now!
	if (rc == 0 && rs == 0) return;

	//Handle the async operation case
	if (rc == 592 )
	{
	    //RS is the job ID.
	    asynchronousCall(String.valueOf(rs));
	    return;
	}
	
	
	//sucessful return codes.
	if(rc == 0)
	{
	    if(rs == 8) throw new ObjectDirectoryOfflineException(desc + "Request was successful, but the object directory was not brought online.");
	}
	else if(rc == 12)
	{
	    throw new SMAPIException(desc + "Output data exceeds the output buffer.");
	}
	else if(rc == 16)
	{
	    if (rs == 0) throw new SMAPIException(desc + "Input data exceeds the input buffer.");
	}
	else if(rc == 24)
	{
	    if(rs == 19)
	    {
		throw new ParmSyntaxErrException(desc + " " + "Parameter value not recognized.");
	    }
	    
	    int errorNumber = rs % 100;
	    int paramNumber = rs / 100;
	    
	    String parmDesc = desc + " " + "Parameter " + paramNumber + ": ";
	    
	    if(errorNumber == 1) parmDesc += "First character of the list name is a semicolon.  This is not allowed.";
	    else if(errorNumber == 2) parmDesc += "Should contain only '0' or '1'.";
	    else if(errorNumber == 10) parmDesc += "Should be a numeric value.";
	    else if(errorNumber == 11) parmDesc += "Unsupported function.";
	    else if(errorNumber == 13) parmDesc += "Value is too long.";
	    else if(errorNumber == 14) parmDesc += "Value is too short.";
	    else if(errorNumber == 15) parmDesc += "Numeric value less than minimum or null value encountered.";
	    else if(errorNumber == 16) parmDesc += "Characters not 0123456789ABCDEF";
	    else if(errorNumber == 17) parmDesc += "Characters not 0123456789ABCDEF-";
	    else if(errorNumber == 18) parmDesc += "Numeric value greater than maximum";
	    else if(errorNumber == 19) parmDesc += "Unrecognized value";
	    else if(errorNumber == 23) parmDesc += "Conflicting parameter specified";
	    else if(errorNumber == 24) parmDesc += "Required parameter is missing";
	    else if(errorNumber == 25) parmDesc += "Extraneous parameter specified";
	    else if(errorNumber == 26) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	    else if(errorNumber == 36) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
	    else if(errorNumber == 37) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789-";
	    else if(errorNumber == 42) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789@#$+-:";
	    else if(errorNumber == 43) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789@#$+-:_";
	    else if(errorNumber == 44) parmDesc += "Characters not ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789@#$+-:_=";
	    else if(errorNumber == 45) parmDesc += "Directory name is not valid";
	    else if(errorNumber == 99) parmDesc += "Non-breaking characters: non-blank, non-null, non-delete, non-line-end, non-carriage return, non-line-feed";
	    else parmDesc += "Undocumented errorNumber: " + errorNumber;
	    
	    throw new ParmSyntaxErrException(parmDesc);
	}
	else if(rc == 36)
	{
	    if(rs == 0) throw new NamelistFileUpdateException(desc + "Namelist file cannot be updated.");
	}
	else if(rc == 100)
	{
	    if(rs == 8) throw new RequestNotAuthorizedException(desc + "Request denied by external security manager.");
	    if(rs == 12) throw new RequestNotAuthorizedException(desc + "Request denied by directory manager.");
	    if(rs == 16) throw new RequestNotAuthorizedException(desc + "Request denied by SMAPI server.");
	}
	else if(rc == 104)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Authorization file not found.");
	}
	else if(rc == 106)
	{
	    if(rs == 0) throw new AuthFileUpdateException(desc + "Authorization file cannot be updated.");
	}
	else if(rc == 108)
	{
	    if(rs == 0) throw new RequestNotAuthorizedException(desc + "Username or password not valid.");
	}
	else if(rc == 112)
	{
	    if(rs == 0) throw new AuthFileEntryDoesNotExist(desc + "Authorization file Entry does not exist.");
	}
	else if(rc == 120)
	{
	    if(rs == 0) throw new RequestNotAuthorizedException(desc + "Username or password not valid.");
	}
	else if(rc == 128)
	{
	    if(rs == 0) throw new PasswordExpiredException(desc + "Password has expired for this user.");
	}
	else if(rc == 188)
	{
	    throw new SMAPIException(desc + "ESM Error: rs=" + rs);
	}
	else if(rc == 192)
	{
	    throw new SMAPIException(desc + "SMAPI Server authentication internal Error: rs=" + rs);
	}
	else if(rc == 196)
	{
	    if(rs == 10) throw new SMAPIException(desc + "Parameter list contains too many parameters.");
	    if(rs == 11) throw new SMAPIException(desc + "Parameter list contains too few parameters.");
	    
	    throw new SMAPIException(desc + "Internal server error: rs=" + rs);
	}
	else if(rc == 200)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Image operation error.");
	    if(rs == 4) throw new ImageNotFoundException(desc + "Guest not found.");
	    if(rs == 8) throw new ImageAlreadyActiveException(desc + "Guest is already active.");
	    if(rs == 12) throw new ImageNotActiveException(desc + "Guest not active.");
	    if(rs == 16) throw new SMAPIException(desc + "This guest is currently being deactivated.  You action was not performed.");
	    if(rs == 24) throw new ListNotFoundException(desc + "The specified namelist was not found");
	    if(rs == 28) throw new SomeImagesNotActivatedException(desc + "Some guests in the list could not be activated.");
	    if(rs == 32) throw new SomeImagesNotDeactivatedException(desc + "Some guests in the list could not be deactivated.");
	    if(rs == 36) throw new SMAPIException(desc + "Deactivate/recycle operatrion is taking longer than expected.");
	}
	else if (rc == 204)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Image device usage error");
	    if(rs == 4) throw new ImageDeviceExistsException("Guest's device already exists");
	    if(rs == 8) throw new ImageDeviceNotFoundException("That guest does not have the specefied device");
	    if(rs == 12) throw new SMAPIException(desc + "Image device is busy.");
	    if(rs == 16) throw new SMAPIException(desc + "Image device is not available.");
	    if(rs == 20) throw new ImageDeviceAlreadyConnectedException(desc + "Image device is already connected.");
	}
	else if(rc == 208)
	{
	    if(rs == 4) throw new ImageDiskInUseException(desc + "Image disk in use.");
	    if(rs == 8) throw new ImageDiskNotInUseException(desc + "Image disk not in use.");
	    if(rs == 12) throw new ImageDiskNotAvailableException(desc + "Image disk is not available.");
	    if(rs == 16) throw new ImageDiskCantBeSharedException(desc + "Image disk cant be shared as requested.");
	    if(rs == 20) throw new ImageDiskAlreadySharedException(desc + "Disk already shared.");
	    if(rs == 28) throw new ImageDiskPasswordException(desc + "A password is needed to access this disk");
	    if(rs == 32) throw new ImageDiskPasswordException(desc + "Incorect disk password given.");
	} 
	else if(rc == 212)
	{
	    if(rs == 4) throw new PartnerImageNotFoundException(desc + "Partner Image not found");
	    if(rs == 8) throw new ImageNotAuthorizedException(desc + "image is not authorized to connect.");
	    if(rs == 12) throw new LANDoesNotExistException(desc + "That Guest lan does not exist");
	    if(rs == 16) throw new ImageNotFoundException(desc + "The guest who suposedly owns this device does not exist.");
	    if(rs == 20) throw new ImageNotActiveException(desc + "The guest specified as LAN owner is not active.");
	    if(rs == 24) throw new LANAlreadyExistsException(desc + "That Guest lan already exists");
	    if(rs == 28) throw new ImageDeviceTypeException(desc + "Image device type error.");
	    if(rs == 32) throw new ImageDeviceNotConnectedException(desc+ "The image device is not connected");
	    if(rs == 36) throw new VSwitchAlreadyExistsException(desc + "That VSwitch already exists");
	    if(rs == 40) throw new VSwitchDoesNotExistException(desc + "That VSwitch does not exist");
	    if(rs == 44) throw new AccessAlreadyGrantedException(desc + "Access already granted to that guest for this network.");
	    if(rs == 48) throw new VSwitchVLANNotFound(desc + "The requested VSwitch does not have the given VLan ID.");
	    if(rs == 96) throw new SMAPIException("Could not connect user to LAN.  Unknown reason.");
	}
	else if(rc == 300)
	{
	    if(rs == 8) throw new DeviceNotFoundException(desc + "That device was not found.");
	    if(rs == 10) throw new DeviceNotAvailableForAttachmentException(desc + "Device not available for attachment.");
	    if(rs == 12) throw new DeviceNotAVolumeException(desc + "This device is not a volume.");
	    if(rs == 14) throw new NoFreeModesException(desc + "No free modes.");
	    if(rs == 16) throw new DeviceVaryOnlineFailedException(desc + "Varying the device online has failed.");
	    if(rs == 18) throw new VolumeLabelNotFoundException(desc + "Volume label not found in system config.");
	    if(rs == 20) throw new VolumeLabelInUseException(desc + "Volume label already in use in system config.");
	    if(rs == 22) throw new SMAPIException(desc + "Parm disks 1 and 2 are the same.");
	    if(rs == 24) throw new SMAPIException(desc + "Error linking Parm disks 1 or 2.");
	    if(rs == 28) throw new SMAPIException(desc + "Parm disks 1 or 2 not RW.");
	    if(rs == 32) throw new SMAPIException(desc + "System config not found on parm disk 1.");
	    if(rs == 34) throw new SMAPIException(desc + "System config has bad data.");
	    if(rs == 36) throw new SMAPIException(desc + "Syntax errors updating sysconfig.");
	    if(rs == 38) throw new SMAPIException(desc + "CP disk modes not available.");
	    if(rs == 40) throw new SMAPIException(desc + "parm disk 1 or 2 is full.");
	    if(rs == 42) throw new SMAPIException(desc + "parm disk 1 or 2 access denied.");
	    if(rs == 44) throw new SMAPIException(desc + "parm disk 1 or 2 missing password.");
	    if(rs == 46) throw new SMAPIException(desc + "parm disk 1 or 2 incorrect password.");
	    if(rs == 48) throw new SMAPIException(desc + "parm disk 1 or 2 is not in server's user directory.");
	    if(rs == 50) throw new SMAPIException(desc + "Error in release of CPRELEASE parm disk 1 or 2.");
	    if(rs == 52) throw new SMAPIException(desc + "Error in access of CPACCESSparm disk 1 or 2.");
	    
	}
	else if(rc == 396)
	{
	    if(rs == 40) throw new SMAPIException(desc + "This could be an error, or it could mean just about anything.");
	    if(rs == 44) throw new SSSSENDFailedException(desc + "SSSSEND has timed out");
	    throw new SMAPIException(desc + "Unknown Internal system error. rs="+rs);
	}
	else if(rc == 400)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Image definition error");
	    if(rs == 4) throw new ImageDefNotFoundException("Could not find that guests directory entry.");
	    if(rs == 8) throw new ImageAlreadyDefinedException("A guest with that name has already been defined.");
	    if(rs == 12) throw new ImageLockedException("Guest is locked");
	    if(rs == 16) throw new SMAPIException(desc + "Guest could not be deleted.");
	    if(rs == 20) throw new PrototypeNotFoundException(desc + "prototype not found.");
	    if(rs == 24) throw new ImageNotLockedException("Image is not locked.");
	}
	else if(rc == 404)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Image device definition error");
	    if(rs == 4) throw new ImageDeviceDefinedException(desc + "Image device already defined");
	    if(rs == 8) throw new ImageDeviceNotDefinedException(desc + "Image device not defined");
	    if(rs == 24) throw new SMAPIException(desc + "Image device type not same as source");
	    if(rs == 28) throw new SMAPIException(desc + "Image device size not same as source");
	}
	else if(rc == 408)
	{
	    if(rs == 4) throw new ImageDiskAlreadyDefinedException(desc + "Image disk already defined.");
	    if(rs == 8) throw new ImageDiskNotDefinedException(desc + "Image disk not defined.");
	    if(rs == 24) throw new ImageSpaceNotAvailableException(desc + "Requested amount of disk space not available.");
	}
	else if(rc == 412)
	{
	    if(rs == 4) throw new PartnerImageNotFoundException(desc + "Partner Image not found");
	    if(rs == 16) throw new ImageDeviceNotConnectedException(desc + "Image device not connected or device could not be disconnected because the given parameters do not match the directory parameters."); 
	}
	else if(rc == 416)
	{
	    if(rs == 0) throw new PrototypeDefinitionException(desc + "Prototype definitiion error.");
	    if(rs == 4) throw new PrototypeNotFoundException(desc + "prototype not found.");
	    if(rs == 8) throw new PrototypeAlreadyExistsException(desc + "Prototype already exists.");
	    if(rs == 16) throw new ImageDiskCantBeSharedException(desc + "Image disk cant be shared as requested.");
	    if(rs == 28) throw new ImageDiskPasswordException(desc + "A password is needed to access this disk");
	    if(rs == 32) throw new ImageDiskPasswordException(desc + "Incorect disk password given.");
	}
	else if(rc == 420)
	{
	    if(rs == 4) throw new DASDGroupRegionVolumeAlreadyDefinedException(desc + "That group, region, or volume has already been defined.");
	    if(rs == 8) throw new DASDGroupRegionVolumeNotDefinedException(desc + "That group, region, or volume his not defined.");
	    if(rs == 12) throw new DASDRegionNameException(desc + "That region name is not included in the group.");
	    if(rs == 36) throw new DASDofflineOrNotDASD(desc + "The requested volume is offline or is not a DASD device.");
	}
	else if(rc == 440)
	{
	    if(rs == 0) throw new PolicyImageException(desc + "Image name violates acceptable naming policy.");
	    if(rs == 4) throw new PolicyImageNameTooLongException(desc + "Image name is too long.");
	    if(rs == 8) throw new PolicyImageNameTooShortException(desc + "Image name is too short.");
	    if(rs == 12) throw new PolicyImageException(desc + "Image name violates acceptable naming policy.");
	}
	else if(rc == 444)
	{
	    if(rs == 0) throw new PolicyPasswordException(desc + "Password violates acceptable naming policy.");
	    if(rs == 4) throw new PolicyPasswordTooLongException(desc + "Password is too long.");
	    if(rs == 8) throw new PolicyPasswordTooShortException(desc + "Password is too short.");
	    if(rs == 12) throw new PolicyPasswordException(desc + "Password violates acceptable naming policy.");
	}
	else if(rc == 448)
	{
	    if(rs == 0) throw new PolicyAccountException(desc + "Account number violates acceptable naming policy.");
	    if(rs == 4) throw new PolicyAccountNumberTooLongException(desc + "Account number is too long.");
	    if(rs == 8) throw new PolicyAccountNumberTooShortException(desc + "Account number is too short.");
	    if(rs == 12) throw new PolicyAccountException(desc + "Account number violates acceptable naming policy.");
	}
	else if(rc == 500)
	{
	    if(rs == 0) throw new SMAPIException(desc + "Directory manager request could nto be completed.");
	    if(rs == 4) throw new SMAPIException(desc + "Directory manager not accepting updates.");
	    if(rs == 8) throw new SMAPIException(desc + "Directory manager not available.");
	    if(rs == 12) throw new SMAPIException(desc + "Directory manager has been disabled.");
	    if(rs == 16) throw new SMAPIException(desc + "Directory manager was interrupted.");
	}
	else if(rc == 504)
	{
	    throw new SMAPIException("Directory manager error: Target ID Not added.  rs="+rs);
	}
	else if(rc == 596)
	{
	    if(rs == 3519) throw new SMAPIException(desc + "Directory manager error: Datamove machines not defined or incorrectly defined.");
	    
	    throw new SMAPIException(desc + "Directory manager error.  rs="+rs);
	}
	else if(rc == 600)
	{
	    if(rs == 8) throw new PageRangeException(desc + "Bad page range specified.");
	    if(rs == 12) throw new ImageNotActiveException(desc + "Guest not active.");
	    if(rs == 16) throw new SaveSegmentException(desc + "Could not save segment.  Unknown reason.");
	    if(rs == 20) throw new RequestNotAuthorizedException(desc + "not authorized for function on this segment.");
	    if(rs == 24) throw new SMAPIException(desc + "Conflicting Parameters.");
	    if(rs == 28) throw new SegmentNotFoundException(desc + "requested segment does not exist.");
	    if(rs == 299) throw new SegmentFileAlreadyExistsException(desc + "Class S (skeleton) segment file already exists.");
	    
	}
	else if(rc == 700)
	{
	    throw new SMAPIException(desc + "Null password entered for untrusted user.");
	}
	else if(rc == 800)
	{
	    throw new SMAPIException(desc + "VMRM Error.  VMRM exceptions have not yet been enumerated.");
	}
	else if(rc == 999)
	{
	    if (rs == 0) throw new SMAPIException(desc + "Function call is unsupported in this version .");
	}
	
	throw new SMAPIException(desc + "Unknown RC/RS pair! rc=" + rc + " rs=" + rs);
    }
}
