/*
 * Created on Jun 30, 2005
 */
package live.dto;

import java.util.Date;

import java.util.Comparator;

import live.threads.ThreadedUserAction;

/**
 * This DTO is NOT runnable, it just holds information about a thread. And
 * actually, it holds information that is specific to a ThreadedUserAction, so
 * it's not generically useful for any thread.
 * 
 * @author Evan Driscoll
 */
public class Thread
{
	public int ID;
	private String description;
	private Date timeEnded;
	private Date timeStarted;
	private boolean finished;
	private String result;
	private int status;
	private boolean isContainer;

	/**
	 * Makes a thread DTO with the specified attributes
	 * 
	 * @param ID The ID of the thread
	 * @param description User-friendly description of what the thread is doing
	 * @param timeStarted
	 * @param timeEnded
	 * @param result User-friendly description of the thread's progress
	 */
	public Thread(int ID, String description, Date timeStarted, Date timeEnded,
			String result, int status)
	{
		this.ID = ID;
		this.setDescription(description);
		this.setTimeStarted(timeStarted);
		this.setTimeEnded(timeEnded);
		finished = !(timeEnded == null);
		setStatus(status);
		isContainer = false;
	}

	/**
	 * Same as {@link #Thread(int, String, Date, Date, String, int)}, but it pulls 
	 * its parameters from a {@link live.threads.ThreadedUserAction}  
	 * 
	 * @param action The action to get the thread's parameters from
	 */
	public Thread(ThreadedUserAction action)
	{
		ID = action.getID();
		setDescription(action.description());
		setTimeStarted(action.getTimeStarted());
		setTimeEnded(action.getTimeFinished());
		finished = action.isFinished();
		setResult(action.result());
		setStatus(action.getStatus());
		isContainer = (action instanceof live.threads.Runner);
	}

	/**
	 * Returns the whether it is a container 
	 *
	 * @return Whether it is a container or not
	 */
	public boolean isContainer()
	{
		return isContainer;
	}
	
	/**
	 * Sets the description
	 *
	 * @param description The description to set.
	 */
	public void setDescription(String description)
	{
		this.description = description;
	}

	/**
	 * Returns the description
	 *
	 * @return Returns the description.
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * Sets the time the task started
	 *
	 * @param timeStarted The timeStarted to set.
	 */
	public void setTimeStarted(Date timeStarted)
	{
		this.timeStarted = (Date) timeStarted.clone();
	}

	/**
	 * Returns the time the task started
	 *
	 * @return Returns the timeStarted.
	 */
	public Date getTimeStarted()
	{
		return (Date) timeStarted.clone();
	}

	/**
	 * Sets the time the task ended
	 *
	 * @param timeEnded The timeEnded to set.
	 */
	public void setTimeEnded(Date timeEnded)
	{
		this.timeEnded = (Date) timeEnded.clone();
	}

	/**
	 * Returns the time the task ended 
	 *
	 * @return Returns the timeEnded.
	 */
	public Date getTimeEnded()
	{
		return (Date) timeEnded.clone();
	}

	/**
	 * Returns whether the task is finished
	 *
	 * @return Returns whether the task is finished.
	 */
	public boolean isFinished()
	{
		return finished;
	}

	/**
	 * Sets the result string to a value
	 *
	 * @param result The result to set.
	 */
	public void setResult(String result)
	{
		this.result = result;
	}

	/**
	 * Returns the result string
	 *
	 * @return the result.
	 */
	public String getResult()
	{
		return result;
	}

	/**
	 * Returns the thread ID
	 *
	 * @return Returns the ID
	 */
	public int getID()
	{
		return ID;
	}

	/**
	 * Returns the ID as an Integer instead of an int. Used for the struts
	 * <bean:write> tags that need objects upon which to do reflection.
	 * 
	 * @return The ID as an Integer
	 */
	public Integer getBoxedID()
	{
		return new Integer(ID);
	}

	/**
	 * Sets the thread's status
	 *
	 * @param status The status to set.
	 */
	public void setStatus(int status)
	{
		this.status = status;
	}

	/**
	 * Returns the thread's status
	 *
	 * @return Returns the status.
	 */
	public int getStatus()
	{
		return status;
	}
	
	////////////////////// COMPARATOR ///////////////////////////
	
	/**
	 * This is a base class for the different comparator styles. See
	 * {@link live.struts.actions.ShowStatusAction} for a use of this
	 * and why it is necessary for clean code there. 
	 */
	abstract public static class ThreadComparator implements Comparator
	{
		private int reverse;

		/**
		 * Makes a comparator that will optionally reverse its decision
		 * 
		 * @param reverseOrder Order in reverse of what it otherwise would?
		 */
		public ThreadComparator(boolean reverseOrder)
		{
			setReverse(reverseOrder);
		}

		/**
		 * Makes a comparator that will not reverse its decision
		 */
		public ThreadComparator()
		{
			this(false);
		}

		/**
		 * Returns this so it can be chained easily, e.g.
		 * foo(myComparator.setReverse(true)). Again, simplifies code.
		 * 
		 * Note that the <tt>reverseOrder</tt> param ignores the current
		 * state of the object. So calling setReverse(true) on a comparator
		 * that is reversed will not make it work in its normal order again.
		 * 
		 * @param reverseOrder Whether to reverse the output
		 * @return The comparator so it can be chained as above
		 */
		public ThreadComparator setReverse(boolean reverseOrder)
		{
			reverse = reverseOrder ? -1 : 1;
			return this;
		}

		/**
		 * This makes sure that o1 and o2 are both threads, calls the abstract
		 * {@link #compare(Thread, Thread)}, and possibly reverses that decision.
		 * 
		 * @param o1 The first object to compare
		 * @param o2 The second object to compare
		 * @return 0 if o1==o2, a negative number if o1<o2, a positve number if o1>o2
		 * 
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		final public int compare(Object o1, Object o2)
		{
			if(o1 instanceof Thread && o2 instanceof Thread)
				// reverse is either -1 or 1, so it will change the sign or not
				return reverse * compare((Thread)o1, (Thread)o2);
			else
				throw new RuntimeException("compare: o1 or o2 not a Thread");
		}

		/**
		 * This does the actual comparison. Note that this should ignore any
		 * reversing option, because {@link #compare(Object, Object)} calls this
		 * and reverses the return itself.
		 * 
		 * @param t1 The first thread to compare
		 * @param t2 The second thread to compare
		 * @return 0 if t1==t2, a negative number if t1<t2, a positve number if t1>t2
		 * 
		 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
		 */
		abstract public int compare(Thread t1, Thread t2);
	}

	/**
	 * This comparator orders threads by the starting value
	 */
	public static class ComparatorByTimeStarted extends ThreadComparator
	{
		/**
		 * Compares two threads by subtracting the time started 
		 * for each
		 * @param t1 First thread
		 * @param t2 Second thread
		 *
		 * @return int Difference in starting times between the two threads
		 */
		public int compare(Thread t1, Thread t2)
		{
			// The cast is because the non-cast type is long. However,
			// the difference will always be relatively small in practice,
			// and an int can hold a time up to 68 years. If this program
			// still runs in 68 years, I want a turing medal. ;)
			return (int)(t1.getTimeStarted().getTime() - t2.getTimeStarted()
					.getTime());
		}
	}

	/**
	 * This comparator orders threads by the ending value. Because not all
	 * threads have an ending value (because they haven't ended and we can't yet
	 * see into the future), will sort all threads that are still ongoing as
	 * less than all threads that have ended. It breaks ties with the time
	 * started.
	 * 
	 * (In other words, sorting with this puts all ongoing threads before all
	 * complete threads, with the ongoing threads sorted by time started and
	 * complete threads by time ended.)
	 */
	public static class ComparatorByTimeEnded extends ThreadComparator
	{
		/**
		 * Compares two threads by subtracting the time ended 
		 * for each
		 * @param t1 First thread
		 * @param t2 Second thread
		 *
		 * @return int Difference in starting times between the two threads, returns
		 *          -1 if thread1 isn't ended and 1 if thread2 isn' ended
		 */
		public int compare(Thread t1, Thread t2)
		{
			if(t1.getTimeEnded() == null && t2.getTimeEnded() == null)
			{
				// Go off of started times
				//The cast is because the non-cast type is long
				return (int)(t1.getTimeStarted().getTime() - t2
						.getTimeStarted().getTime());
			}
			else if(t1.getTimeEnded() == null)
			{
				// Null comes first, so it is less than any non-null
				return -1;
			}
			else if(t2.getTimeEnded() == null)
			{
				// Non-null comes after any null
				return 1;
			}
			else
			{
				// The cast is because the non-cast type is long, and it's safe
				// because no difference will be more than 68 years
				return (int)(t1.getTimeEnded().getTime() - t2.getTimeEnded()
						.getTime());
			}
		}
	}
	/**
	 * This comparator orders threads by their descriptions
	 */
	public static class ComparatorByDescription extends ThreadComparator
	{
	       /**
		 * Compares two threads by comparing the descriptions
		 * @param t1 First thread
		 * @param t2 Second thread
		 *
		 * @return int Difference in descriptions between the two threads
		 */
		public int compare(Thread t1, Thread t2)
		{
			return t1.getDescription().compareTo(t2.getDescription());
		}
	}

	/**
	 * This comparator orders threads by their descriptions
	 */
	public static class ComparatorByResult extends ThreadComparator
	{
		/**
		 * Compares two threads by their results
		 * @param t1 First thread
		 * @param t2 Second thread
		 *
		 * @return int Difference in results between the two threads
		 */
		public int compare(Thread t1, Thread t2)
		{
			return t1.getResult().compareTo(t2.getResult());
		}
	}
}
