/**
 * Network.java
 * Created on June 13, 2005, 8:19 AM
 */

package live.dto;

import java.util.LinkedList;
import java.util.Collection;

/**
 * Network is used by the cache to store information about a specific network.
 * 
 * @author Jason J. Herne
 */
public class Network
{
    public static final int NETWORK_GLAN = 0;
    public static final int NETWORK_VSWITCH = 1;
    
    private String networkName;
    private int networkType;        // 0=GLAN, 1=VSWITCH, as above
    
    public String owner;         //GLAN, VSWITCH
    public String type;	         //GLAN
    public int numConnected;     //GLAN, VSWITCH
    public int maxConnected;     //GLAN, VSWITCH
    public boolean restricted;   //GLAN, VSWITCH
    public boolean priRouter;    //VSWITCH
    public int ipTimeout;        //VSWITCH
    public int queueStorLim;     //VSWITCH
    public String rdev1;         //VSWITCH
    public String rdev2;         //VSWITCH
    public String rdev3;         //VSWITCH
    public String portname1;     //VSWITCH
    public String portname2;     //VSWITCH
    public String portname3;     //VSWITCH
    public LinkedList members;	 //Only for a VSWITCH right now
    
    /**
     * Creates a Network and sets its name and type.
     * 
     * @param	networkName The name of this network
     * @param	networkType The type of this network
     */
    public Network(String networkName, int networkType)
    {
	setNetworkName(networkName);
	setNetworkType(networkType);
	members = new LinkedList();
    }
    
    /**
     * Sets the name of this network.
     *
     * @param	networkName The network name to use.
     */
    public void setNetworkName(String networkName)
    {
	this.networkName = networkName;
    }
    
    /**
     * Gets the name of this network.
     *
     * @return Name of this network.
     */
    public String getNetworkName()
    {
	return networkName;
    }
    
    /**
     * Sets the type of this network.  This can be 0 for GLAN or 1 for VSWITCH.
     * The defined constants NETWORK_GLAN and NETWORK_VSWICH can be used as well.
     *
     * @param	networkType The network type to set. 
     */
    public void setNetworkType(int networkType)
    {
	this.networkType = networkType;
    }
    
    /**
     * Gets the type of this network.
     *
     * @return type of this network.
     */
    public int getNetworkType()
    {
	return networkType;
    }
    
    /**
     * Adds a member to the network
     * 
     * @param guestName Name of the guest to add
     */
    public void addMember(String guestName)
    {
    	members.add(guestName);
    }
    
    /**
     * Returns the members of the network (a Collection of guest names
     * as Strings)
     * 
     * @return The members of the network
     */
    public Collection getMembers()
    {
    	return members;
    }
    
    /**
     * Returns a nice, readable version of the network in the form:
     * [Network (network name): [member, member, ...]]
     * 
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
    	return "[Network " + getNetworkName() + ": " + getMembers().toString() + "]";
    }
}
