/*
 * Group.java Created on June 13, 2005, 7:08 AM
 */
package live.dto;

import java.util.LinkedList;
import java.util.ListIterator;

/**
 * Group is used by the cache to store information about a specific group.
 * 
 * @author Jason J. Herne
 */
public class Group implements Comparable
{
	/**
	 * A constant that defines a placeholder for a group that is actually
	 * (ungrouped). The group that represents the ungrouped pseudogroup 
	 * will have this as a name. This should be the only place you should
	 * get this string! There are a couple places where the "ungroupedness"
	 * is tested a name using == instead of equals. Always using this very
	 * string will allow this (substantially easier and faster) test to
	 * suffice, but if you use another string, even if it is "equal" to
	 * <tt>UNGROUPED_NAME</tt>, stuff will break.
	 */
	public static final String UNGROUPED_NAME = "[ungrouped]";

	private String groupName;

	private LinkedList members;

	/**
	 * Creates a Group and sets its name.
	 * 
	 * @param groupName The name of this group
	 */
	public Group(String groupName)
	{
		members = new LinkedList();
		setGroupName(groupName);
	}

	/**
	 * Gets the name of this group.
	 * 
	 * @return Name of this group.
	 */
	public String getGroupName()
	{
		return groupName;
	}

	/**
	 * Sets the name of this group.
	 * 
	 * @param groupName The group name to use.
	 */
	public void setGroupName(String groupName)
	{
		this.groupName = groupName;
	}

	/**
	 * Adds the name of a guest to this group.
	 * 
	 * @param guestNameToAdd The name of the guest to add to this group.
	 */
	public void addMember(String guestNameToAdd)
	{
		members.add(guestNameToAdd);
		java.util.Collections.sort(members);
	}

	/**
	 * Returns the index of guestNameToFind or -1 if it is not present
	 * 
	 * @param guestNameToFind The name of the guest we are looking for in the
	 *            list.
	 * @return The index of <tt>guestNameToFind</tt> or -1
	 */
	public int findMember(String guestNameToFind)
	{
		int size = members.size();

		for(int i = 0 ; i < size ; i++)
		{
			if(guestNameToFind.equals(members.get(i)))
				return i;
		}

		return -1;
	}

	/**
	 * Removes the guest from this group given by the parameter.
	 * 
	 * @param guestNameToRemove The guest to remove from this group.
	 */
	public void removeMember(String guestNameToRemove)
	{
		int index = findMember(guestNameToRemove);
		if(index != -1)
			members.remove(index);
	}

	/**
	 * Removes the member of the guest list given by the index. NOTE: If you
	 * plan on removing a guest from the groups list with this function, you
	 * MUST be careful! Another thread could have added/removed a guest between
	 * the time that you looked up the index you want to delete and the time
	 * that you called this function!! Be ABSOLUTELY SURE that your threads lock
	 * on this list object for the duration of any lookup/change events.
	 * 
	 * @param indexOfMemberToRemove Index of the guest we wish to remove from
	 *            the list.
	 */
	public void removeMember(int indexOfMemberToRemove)
	{
		if(indexOfMemberToRemove >= 0 && indexOfMemberToRemove < members.size())
			members.remove(indexOfMemberToRemove);
	}

	/**
	 * Returns a {@link java.util.ListIterator} that can be used to get every guest name that is a
	 * member of this group. Be careful of synchronization, especially as
	 * THIS CLASS DOESN'T.
	 * 
	 * @return The {@link java.util.ListIterator} used to iterate over this list.
	 */
	public ListIterator getMembers()
	{
		return members.listIterator();
	}

	/**
	 * Returns the whole member list. Again, careful of synchronization.
	 * 
	 * @return The {@link java.util.LinkedList} of members.
	 */
	public LinkedList getMemberList()
	{
		return members;
	}

	/**
	 * This allows groups to be ordered by the guest name, so if you have
	 * a sorted collection (for instance, and this is by far the motivation)
	 * the groups will be in alphabetical order.
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o)
	{
		Group g = (Group)o;

		return getGroupName().compareTo(g.getGroupName());
	}
}
