package live.cache;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextListener;
import javax.servlet.ServletContextEvent;

import org.apache.struts.util.MessageResources;

import java.util.Date;
import live.constants.Properties;
import live.constants.Attributes;

/**
 * This class starts a background thread on the server that keeps the cache
 * updated. The cache is accessable from the
 * live.constants.Attributes.servletContext.cache attribute of the
 * ServletContext, and this thread is accessable from the
 * Attributes.servletContext.cacheThread attribute. This object needs to be
 * accessable so that the user can trigger cache updates from a refresh button.
 * 
 * @author Evan Driscoll
 */
/*
 * TODO: Change the update time so it gets it from the config file, and fix the
 * bit with getting the server info from the config file. (Now it pulls the
 * information from an application scoped MessageResources object. Problem is,
 * this object is maintained by Struts and isn't created until after this
 * contextlistener is called. The probable solution is to make it so that login
 * actions schedule a cheche update, at least if one isn't already present.)
 * 
 * TODO: In retrospect, it probably would have made more sense to implement this
 * and Cache as singletons. In re-retrospect, this is not true with the cache
 * because the CacheThread creates a new instance every update. In re-re-retrospect
 * if the Cache were broken up into modules for guests, nets, and segments with
 * Cache being an aggregate of those and providing an interface to them, then
 * Cache could (should?) be made a singleton.
 * 
 * You'd probably have to get rid of the ServeltContextListener bit of the 
 * CacheThread if it's made a singleton though, because then the server wouldn't
 * be able to create it. But that's okay I think, because we no longer can
 * do an update on server start anyway. (A slick solution would be to make another
 * class that wasn't a singleton but was a ServletContextListener whose init
 * method would get the instance of CacheThread and update it.)
 */
public class CacheThread extends Thread implements ServletContextListener
{
	/** The number of milliseconds in a second */
	private static final long SECONDS = 1000;

	/** The number of milliseconds in a minute */
	private static final long MINUTES = 60 * SECONDS;

	/** The number of milliseconds in an hour */
	private static final long HOURS = 60 * MINUTES;

	/**
	 * This is the number of milliseconds before the cache will automatically
	 * update.
	 * 
	 * TODO: Get this fnom a config file, and provide separate frequencies for
	 * the different updates.
	 */
	public static final long UPDATE_DELTA_MS = 10 * HOURS;

	/**
	 * This is how long the thread sleeps before waking up and checking if it is
	 * supposed to do an update. Note that this is distinct from
	 * UPDATE_DELTA_MS, which is how often an update is supposed to happen. This
	 * should be low so that if the user presses refresh he or she sees an
	 * update shortly after, because it could potentially be SLEEP_TIME_MS
	 * milliseconds from when they press the button to when the update even
	 * starts.
	 */
	public static final long SLEEP_TIME_MS = 1 * SECONDS;

	// Used to get to the MessageResources object
	private ServletContext context;

	/** The time the nextUpdate is scheduled for */
	private Date nextUpdate;

	/** Tells whether the thread should be running */
	private boolean continueRunning = true;

	/**
	 * This function enters an infinite loop that checks for new updates for the
	 * server ever so ofter.
	 * 
	 * TODO: A possible idea for future expansion (not likely during our time
	 * here) is to provide a more flexible method of scheduling updates so you
	 * could do, for example, "update every 5 minutes from 9am to 5pm, 10
	 * minutes from 7am to 9am and 5pm to 8pm, and 20 minutes from 8pm to 7am",
	 * or even more complex functions, through the use of policy classes. This
	 * would be a fairly large chunk of coding for relatively little benefit, so
	 * it's low on the priority ladder.
	 */
	public void run()
	{
		context.setAttribute(Attributes.servletContext.cacheThread, this);
		
		// continueUpdates is unset when the thread should die, i.e. when
		// the servlet context is being unloaded. Otherwise, depending on
		// the JVM/server being used, the thread won't die and the server
		// won't die.
		while(continueRunning)
		{
			doUpdateCache();

			// new Date() is the current time since there're no construction
			// parameters. Thus the condition below means "while the next update
			// is after the current time"
			while(nextUpdate.after(new Date()))
			{
				try
				{
					sleep(SLEEP_TIME_MS);
				}
				catch(InterruptedException e)
				{}
			}
		} // while(continueUpdates)
	}

	/**
	 * Called automatically when the server starts. Simply stores the context
	 * and starts the update thread
	 * 
	 * @param event 
	 */
	public void contextInitialized(ServletContextEvent event)
	{
		context = event.getServletContext();
		start();
	}

	/**
	 * Called automatically when the server ends. We tell the thread to die.
	 * At least soon.
	 * 
	 * @param event 
	 */
	public void contextDestroyed(ServletContextEvent event)
	{
		continueRunning = false;
	}

	/**
	 * This function updates the cache. More precisely (and accurately), it
	 * schedules an update for now, so that when the server wakes up again in
	 * something less than SLEEP_TIME_MS, it will update.
	 */
	public void scheduleUpdate()
	{
		scheduleUpdate(new Date());
	}

	/**
	 * This function schedules the next cache update for the specified
	 * parameter. This is true even if <tt>nextUpdate</tt> is after the
	 * currently scheduled update time.
	 * 
	 * @param nextUpdate The time to start the next cache update, within
	 *            SLEEP_TIME_MS
	 */
	public void scheduleUpdate(Date nextUpdate)
	{
		Cache cache = (Cache) context.getAttribute(Attributes.servletContext.cache);
		if(cache != null)
		{
			cache.setState(Cache.UPDATING);
		}
		
		this.nextUpdate = (Date) nextUpdate.clone();
	}
	
	Exception exception = null;
	
	/**
	 * Returns the exception thrown
	 * 
	 * @return The exception
	 */
	public Exception getError()
	{
		return exception;
	}

	/////////////////////// PRIVATE FUNCTIONS ////////////////////////////

	/**
	 * This function actually carries out the cache update. It makes a new cache
	 * object, updates the cache, then sets the cache object in application
	 * scope.
	 * 
	 * The reasoning behind creating a new cache object instead of getting the
	 * current one is that it likely helps to reduce concurrency issues. Rather
	 * than have the cache worry about what happens if someone querys it while
	 * it's updating, we simply use a different object. (I'm not saying the
	 * cache is free from concurrency issues because of this, just that they are
	 * a bit less of a problem.)
	 * 
	 * TODO: Fix the getting of the server parameters
	 */
	private void doUpdateCache()
	{
		exception = null;
		
		// We get the resources object from application scope
		MessageResources resources = (MessageResources)context
				.getAttribute(org.apache.struts.Globals.MESSAGES_KEY);

		String hostName;
		int port;
		String adminName;
		String adminPass;

		if(resources == null)
		{
			long now = (new Date()).getTime();
			nextUpdate = new Date(now + UPDATE_DELTA_MS);
			return;
		}
		else
		{
			hostName = resources.getMessage(Properties.live.host);
			port = Integer.parseInt(resources.getMessage(Properties.live.port));
			adminName = resources.getMessage(Properties.live.adminUser);
			adminPass = resources.getMessage(Properties.live.adminPass);
		}

		live.cache.Cache cache = new live.cache.Cache(hostName, port,
				adminName, adminPass);

		try
		{
			cache.updateCacheFromServer();
			
			Cache ccache = (Cache) context.getAttribute(Attributes.servletContext.cache);
			if(ccache == null)
			{
				ccache = new live.cache.Cache(hostName, port,
						adminName, adminPass);
			
				context.setAttribute(Attributes.servletContext.cache, ccache);
			}
			
			cache.swap(ccache);
		}
		catch(live.cache.CacheException e)
		{
			// "Log" the error
			System.out.println("Cache update error: " + e);
			exception = e;
		}

		// And schedule the next update for UPDATE_DELTA_MS from now
		long now = (new Date()).getTime();
		nextUpdate = new Date(now + UPDATE_DELTA_MS);
	}
}