/*
 * Cache.java Created on June 13, 2005, 11:49 AM
 */
package live.cache;

import java.util.Set;
import java.util.TreeMap;
import java.util.Map;
import java.util.LinkedList;
import java.util.ListIterator;
import java.util.Iterator;

import live.zvmtasks.ZVMTasks;
import live.zvmtasks.ZVMTasksException;
import live.dto.Guest;
import live.dto.Network;
import live.dto.Group;
import live.dto.SavedSegment;

/**
 * Cache uses live's DTOs to cache data obtained from the z/VM server. Updates
 * to the data within (e.g. calling #addGuest(Guest)) does not update anything 
 * on the server. This data is meant to be a model of what is on the z/VM 
 * instance being managed. This data should be updated to reflect successful 
 * changes made to the actual server.
 * 
 * The cache is the source of data for the pages listing guests/group (except
 * for marking active and inactive guests), networks, and (later) shared storage.
 * Other pages, such as member lists for networks and information about guests,
 * is queried live when those pages are loaded.
 * 
 * @author Jason J. Herne
 * @author Evan Driscoll (substantial changes/rewrites)
 */
public class Cache
{
	public static final int VALID = 0;
	public static final int UPDATING = 1;
	public static final int ERROR = 2;
	public static final int UNSET = 3;
	private int state = UNSET;
	
	public int getState() { return state; }
	protected void setState(int newState) { state = newState; }
	
	
	/** Maps cached group names to a group DTO */
	Map groupList = new TreeMap();
	
	/** Maps cached guest names to a guest DTO */
	Map guestList = new TreeMap();

	/** Stores the cached networks as network DTOs */
	LinkedList networkList = new LinkedList();
	
	/* Stores the cached stored segments as network DTOs */
	LinkedList sharedStorageList = new LinkedList();

	/** This is the tasks object used when updating the cache from the server */
	ZVMTasks tasks;

	
	/**
	 * Creates a new instance of Cache that will connect to <tt>hostName</tt>
	 * on port <tt>port</tt> with the specified <tt>userName</tt> and
	 * <tt>password</tt>.
	 */
	public Cache(String hostName, int port, String userName, String password)
	{
		tasks = new ZVMTasks(hostName, port, userName, password);
	}

	/**
	 * Adds a guest to the cache and caches his group name as <tt>groupName</tt>.
	 * If <tt>groupName</tt> is null, adds the guest to <em>ungrouped</em>.
	 * If the guest is already present in the cache, throws a
	 * {@link CacheException} saying so.
	 * 
	 * @param guestName The name of the new cached guest entry.
	 * @param groupName The group to associate with the guest in the new entry.
	 */
	public void addGuest(String guestName, String groupName)
	{
		/* This is probably a good idea but could break current code.
		 * 
		 * if(guestList.containsKey(guestName)) throw new
			CacheException(guestName + " is already in the cache");
			*/

		// Keep the guestList in sync
		Guest guest = new Guest(guestName, groupName);
		guestList.put(guestName, guest);

		// Keep the groupList in sync
		if(groupName == null)
			groupName = Group.UNGROUPED_NAME;

		Group group = null;
		try
		{
			group = (Group)groupList.get(groupName);
		}
		catch(NullPointerException e)
		{
			// This is okay, it just means that groupList doesn't
			// contain groupName. At least one kind of set will throw
			// in the case you try to get something that isn't there.
			
			// group is still null, so we're good. That's what we want.
		}

		// The guest belongs to a group we didn't know about before
		if(group == null)
		{
			group = new Group(groupName);
			groupList.put(groupName, group);
		}

		group.addMember(guestName);
	}

	/**
	 * Adds a guest to the cache and leaves it ungrouped. This means the group
	 * entry for this guest will be <tt>null</tt>.
	 * 
	 * @param guestName The name of the new cached guest entry.
	 */
	public void addGuest(String guestName)
	{
		addGuest(guestName, null);
	}

	/**
	 * Removes a guest from the cached list.
	 * 
	 * No checking that <tt>guest</tt> is actually in the cache.
	 * 
	 * @param guest The guest object to remove from the cached lists.
	 */
	public void removeGuest(Guest guest)
	{
		// Remove the guest from the guestList
		guestList.remove(guest.getGuestName());

		// Remove the guest from its group's list
		String groupName = guest.getGroupName();
		if(groupName == null)
			groupName = Group.UNGROUPED_NAME;

		Group group = (Group)groupList.get(groupName);
		group.removeMember(guest.getGuestName());

		// If that was the last guest in that group, remove the group
		if(group.getMemberList().size() == 0)
			groupList.remove(groupName);
	}

	/**
	 * Removes a guest (given by its name) from the cached list. If the guest is
	 * not found, nothing happens.
	 * 
	 * @param guestName The name of the guest to remove from the cached lists.
	 */
	public void removeGuest(String guestName)
	{
		try
		{
			Guest guest = (Guest)guestList.get(guestName);
			
			if(guest != null)
				removeGuest(guest);
		}
		catch(NullPointerException e)
		{
			// Again, this is OK; some Sets throw if you try to get something
			// that isn't in it. We don't want to propogate this exception,
			// we just do nothing.
		}
	}

	/**
	 * Change the group that a cached guest belongs to. If we do not find that
	 * guest, nothing happens.
	 * 
	 * @param guestName The name of the guest we wish to change.
	 * @param newGroupName The name we wish to set as this guests group.
	 */
	public void changeGuestsGroup(String guestName, String newGroupName)
	{
		// Declare this outside of the try. Grrr...
		Guest guest;
		
		try
		{
			guest = (Guest)guestList.get(guestName);
		}
		catch(Exception e)
		{
			// Again, OK. It just means that guestName wasn't found.
			return;
		}

		if(guest != null)
		{
			String oldGroupName = guest.getGroupName();
			if(oldGroupName == null)
				oldGroupName = Group.UNGROUPED_NAME;

			// Keep the guest up to date
			guest.setGroupName(newGroupName);

			// Keep the old group up to date
			Group group = (Group)groupList.get(oldGroupName);
			group.removeMember(guestName);

			// If that was the last guest in the group, remove the group
			if(group.getMemberList().size() == 0)
				groupList.remove(oldGroupName);

			// Keep the new group up to date
			if(newGroupName == null)
				newGroupName = Group.UNGROUPED_NAME;
			try
			{
				group = (Group)groupList.get(newGroupName);
			}
			catch(NullPointerException e)
			{
				// Again, some Sets will throw if you try to get something
				// not in them. This just means that newGroupName wasn't found
			}

			// The guest belongs to a group we didn't know about before
			if(group == null)
			{
				group = new Group(newGroupName);
				groupList.put(newGroupName, group);
			}

			// Finally we can tell the guest's new group about him
			group.addMember(guestName);
			
		} // if guest != null
	}

	/**
	 * Get an {@link java.lang.ListIterator} that can be used to visit all
	 * guests in the cache.
	 * 
	 * @return {@link java.lang.ListIterator}that can be used to visit all
	 *         guests in the cache.
	 */
	public ListIterator getGuests()
	{
		// Note: you can't just copy the entrySet by passing it as a construction
		// parameter, because it is a set of Map.Entrys, not Guest DTOs.
		LinkedList ret = new LinkedList();
		
		Set set = guestList.entrySet();
		Iterator iter = set.iterator(); 

		while(iter.hasNext())
			ret.add(((Map.Entry)iter.next()).getValue());
			
		return ret.listIterator();
	}

	/**
	 * Remove all guests and groups from the cache
	 */
	private void clearGuestList()
	{
		guestList.clear();

		groupList.clear();
		
		// The ungrouped "group" isn't really a group, and therefore
		// always exists because it doesn't actually exist. Yeah. 
		// Anyway, we must make it.
		Group ungrouped = new Group(Group.UNGROUPED_NAME);
		groupList.put(Group.UNGROUPED_NAME, ungrouped);
	}

	/**
	 * Returns a LinkedList holding names (as Strings) of all groups in
	 * the cache.
	 * 
	 * @return The list of groups in the cache as Strings.
	 */
	public LinkedList getGroupNames()
	{
		// Here we can't pass keyset in as a consturction paramater because
		// we have special handling for :UNGROUPED:, i.e. don't add it.
		// If we constructed it this way, we'd have to go through and find
		// then remove :UNGROUPED:.
		LinkedList ret = new LinkedList();

		// The group names are simply the keys in the group map, so we
		// simply get all of them
		Iterator iter = groupList.keySet().iterator();
		
		while(iter.hasNext())
		{
			String group = (String)iter.next();
			
			// == should work here, because the only way they should get
			// the string :UNGROUPED: is if they get it from the UNGROUPED_NAME
			// constant. Thus all references will be to the same object. 
			if(group != Group.UNGROUPED_NAME)
				ret.add(group);
		}

		return ret;
	}

	/**
	 * Returns a list of Group objects corresponding to all groups in the cache.
	 * (The group objects will know about their group members.)
	 * 
	 * @return List of groups in the cache, as Group objects.
	 */
	public LinkedList getGroups()
	{
		// Same as with getGuests, we can't just copy entrySet in the constructor
		LinkedList ret = new LinkedList();

		Set mappings = groupList.entrySet();

		Iterator iter = mappings.iterator();
		while(iter.hasNext())
		{
			Map.Entry entry = (Map.Entry)iter.next();
			ret.add(entry.getValue());
		}

		return ret;
	}

	/**
	 * Gets all guest objects corresponding to the guests in a given group.
	 * If groupName isn't found, returns an empty list.
	 * 
	 * @param groupName The group to search for.
	 * @return List containing all guests in the given group (as DTO objects)
	 */
	public LinkedList getGuestsInGroup(String groupName)
	{
		Group group;

		try
		{
			group = (Group)groupList.get(groupName);
		}
		catch(Exception e)
		{
			// Some Sets throw if groupName isn't in them
			// We can't put the following if(group == null) in here though,
			// because not all do.
			group = null;
		}

		if(group == null)
		{
			// The group they specified doesn't exist. We don't throw an
			// exception, we return an empty list.
			return new LinkedList();
		}
		else
		{
			// Here we can't copy the collection in the constructor
			// because we go to the guestList.
			LinkedList ret = new LinkedList();

			Iterator iter = group.getMemberList().iterator();
			while(iter.hasNext())
				ret.add(guestList.get(iter.next()));

			return ret;
		}
	}
	
	/**
	 * Gets all guest names corresponding to the guests in a given group. If
	 * groupName isn't found, returns an empty list.
	 * 
	 * @param groupName The group to search for.
	 * @return List containing all guest names in the given group (as
	 *         <tt>String</tt>s)
	 */
	public LinkedList getGuestNamesInGroup(String groupName)
	{
		Group group;

		try
		{
			group = (Group)groupList.get(groupName);
			return group.getMemberList();
		}
		catch(NullPointerException e)
		{
			// Either* the get threw because groupName wasn't found, or
			// group.getMemberList threw because group was null because
			// groupName wasn't found. Either way, groupName wasn't found,
			// so we return an empty list.
			//
			// * Depending on the type of set.
			return new LinkedList();
		}
	}

	/**
	 * Gets all guest objects that are not in a group.
	 * 
	 * @return List containing all guests in the given group (as DTO objecs).
	 */
	public LinkedList getUngroupedGuests()
	{
		return getGuestsInGroup(Group.UNGROUPED_NAME);
	}

	/**
	 * Gets a specific {@link live.dto.Guest} object
	 * 
	 * @param name The name of the guest to return
	 * @return The {@link live.dto.Guest} of that guest
	 */
	public Guest getGuest(String name)
	{
		try
		{
			return (Guest) guestList.get(name);
		}
		catch(NullPointerException e)
		{
			return null;
		}
	}

	/**
	 * Adds a network to the cache by name and type
	 * 
	 * @param networkName The name of the network
	 * @param networkType The type of the network (see {@link live.dto.Network}
	 *            for constants you'll probably want to use)
	 */
	public void addNetwork(String networkName, int networkType)
	{
		networkList.add(new Network(networkName, networkType));
	}

	/**
	 * Adds a network to the cache
	 * 
	 * @param net The network to add
	 */
	public void addNetwork(Network net)
	{
		networkList.add(net);
	}

	/**
	 * Removes the network with the given name from the cache. If the network
	 * isn't found, nothing happens.
	 * 
	 * @param networkName The name of the network to remove
	 */
	public void removeNetwork(String networkName)
	{
		Network curNet;

		// We can't just use networkList.remove
		ListIterator itr = networkList.listIterator();
		while(itr.hasNext())
		{
			curNet = (Network)itr.next();
			if(curNet.getNetworkName().equals(networkName))
			{
				itr.remove();
				return;
			}
		}
	}

	/**
	 * Returns an iterator that can be used to visit all of the networks in the cache.
	 * 
	 * @return An iterator into the Cache's network collection
	 */
	public ListIterator getNetworks()
	{
		return networkList.listIterator();
	}

	/**
	 * Remove all the networks from the cache
	 */
	private void clearNetworks()
	{
		networkList.clear();
	}

	/**
	 * Add a saved segment to the cache
	 *
	 * @param ssName Name of the saved segment to add
	 */
	public void addSavedSegment(String ssName)
	{
		sharedStorageList.add(new SavedSegment(ssName, null, null));
	}

	/**
	 * Remove a saved segment from the cache
	 *
	 * @param ssName Name of the saved segment to remove
	 */
	public void removeSavedSegment(String ssName)
	{
		SavedSegment curSharedStorage;

		ListIterator itr = sharedStorageList.listIterator();
		while(itr.hasNext())
		{
			curSharedStorage = (SavedSegment)itr.next();
			if(curSharedStorage.getName().equals(ssName))
			{
				itr.remove();
				return;
			}
		}
	}

	/**
	 * Returns an iterator that can be used to see all the saved 
	 *  segments in the cache
	 *
	 * @return {@link java.util.ListIterator} An iterator for the cache's saved segment collection 
	 */
	public ListIterator getSavedSegments()
	{
		return sharedStorageList.listIterator();
	}

	/**
	 * Remove all shared storage objects from the shared storage list.
	 */
	public void clearSavedSegments()
	{
		sharedStorageList.clear();
	}

	/**
	 * Removes all cached data from the cache.
	 */
	private void clearCache()
	{
		clearGuestList();
		clearNetworks();
		clearSavedSegments();
	}

	/**
	 * Refreshes the cache so that it is consistent with the server
	 */
	public void updateCacheFromServer() throws CacheException
	{
		try
		{
			state = UPDATING;
			
			clearCache();
			updateGuestsGroupsFromServer();
			updateNetworksFromServer();
			updateSharedStorageFromServer();
			
			state = VALID;
		}
		catch(CacheException e)
		{
			state = ERROR;
			throw e;
		}

	}

	/** ********************* PRIVATE ********************** */

	/**
	 * Refreshes the list of guests and groups from the server.
	 */
	private void updateGuestsGroupsFromServer() throws CacheException
	{
		LinkedList guests;
		LinkedList groups;

		// First, we go to the server and get the data
		try
		{
			guests = tasks.queryAllGuestNames();
			groups = tasks.getGroups();
		}
		catch(ZVMTasksException e)
		{
			throw new CacheException(e);
		}

		// Now, we interpret it.

		// The first step is to simply move all the guests
		// into our map. This is necessary because the queryAllGuestNames
		// just gives us strings, and we need to turn them into actual
		// Guest DTO objects.
		addGuestsToList(guests);

		// Now we go through the groups list and do the same thing. For each
		// group
		// we also tell all the guests in that group what group they belong to.
		addGroupsToList(groups);

		// Finally, we have to maintain our ungrouped list, so we see what
		// guests
		// don't yet have a group
		addUngroupedGuestsToUngrouped();
	}

	/**
	 * Adds everyone in <tt>guests</tt> to the cache. All of the guests will
	 * be added as ungrouped.
	 * 
	 * @param guests The list of guests to add to the cache
	 */
	private void addGuestsToList(LinkedList guests)
	{
		Iterator iter = guests.iterator();
		while(iter.hasNext())
		{
			// For now, there is no group information associated with it
			Guest guest = new Guest((String)iter.next(), null);
			guestList.put(guest.getGuestName(), guest);
		}
	}

	/**
	 * Adds all of the groups in <tt>groups</tt> to the cache, while at the
	 * same time updating all of the guests in each so that they know about what
	 * group they are in.
	 * 
	 * If there is a group with a member which the cache doesn't know about
	 * (i.e. it's not in the guest list), this assumes that you don't want it
	 * in the group and so tries to remove it server side in addition to not
	 * adding it here.
	 * 
	 * This function takes ownership of each of the objects in groups.
	 * 
	 * @param groups The groups to add to the cache (as {@link live.dto.Group}
	 *            s)
	 */
	private void addGroupsToList(LinkedList groups)
	{
		Iterator iter = groups.iterator();
		while(iter.hasNext()) // foreach(group in groups)
		{
			Group group = (Group)iter.next();
			String groupName = group.getGroupName();

			//Group group = new Group(groupName);

			// Now, tell all the guests that are in this group that they
			// are in this group. (They don't know this yet.)
			Iterator memberItr = group.getMembers();
			while(memberItr.hasNext())
			{
				String guestName = (String)memberItr.next();

				try
				{
					Guest guest = (Guest)guestList.get(guestName);
					guest.setGroupName(groupName);

					//group.addMember(guestName);
				}
				catch(Exception e)
				{
					// If this happens, probably the guest was deleted
					// without being removed from the name list. In
					// this case, we try to remove that guest from the
					// name list, as well as the cached list.
					
					// This removes the element that was last returned by
					// next(), not the location currently pointed to. 
					// A very poor name (say, removePrevious() would be 10
					// times better, but exactly what we want in any case.
					memberItr.remove();
					
					// Now remove them from the server.
					System.out.println("Guest " + guestName
							+ " not found. Removing from name list.");

					try
					{
						tasks.setGuestsGroup(guestName, groupName, null);
						System.out.println("Removal successful");
					}
					catch(Exception e2)
					{
						System.out
								.println("Removal failed: " + e2.getMessage());
					}
				}
			} // while(member.hasNext) -- foreach(member in group.getMembers())

			// It's possible that all the guests in a group were deleted
			// without being removed, and the exception catch above that
			// removes such guests from their list was triggered for all
			// guests in the current list (which is empty), so we don't
			// want to store the group because it doesn't exist on the server.
			if(group.getMemberList().size() > 0)
				groupList.put(groupName, group);

		} // while(groupItr.hasNext()) -- foreach(group in groups)
	}

	/**
	 * This function goes through the list of guests and adds any that do not
	 * belong to a group to the member list of the ungrouped list.
	 */
	private void addUngroupedGuestsToUngrouped()
	{
		Group ungrouped = (Group)groupList.get(Group.UNGROUPED_NAME);
		Set mappings = guestList.entrySet();

		Iterator iter = mappings.iterator();
		while(iter.hasNext())
		{
			Map.Entry entry = (Map.Entry)iter.next();
			Guest guest = (Guest)entry.getValue();
			
			// The above 5 lines are essentilly
			// foreach(guest stored in the guest list)

			if(guest.getGroupName() == null)
			{
				String guestName = guest.getGuestName();
				ungrouped.addMember(guestName);
			}
		}
	}

	/**
	 * Refreshes the network list from the server
	 * 
	 * @throws CacheException If there's a problem. (Use getCause() to determine
	 *             what the problem actually is.)
	 */
	private void updateNetworksFromServer() throws CacheException
	{
		LinkedList networks;

		try
		{
			networks = tasks.queryAllNetworks();
		}
		catch(ZVMTasksException e)
		{
			throw new CacheException(e);
		}

		ListIterator itr = networks.listIterator();
		while(itr.hasNext())
		{
			addNetwork((Network) itr.next());
		}
	}

	/**
	 * Refreshes the saved segment list from the server
	 * 
	 * @throws CacheException If there's a problem. (Use getCause() to determine
	 *             what the problem actually is.)
	 */
	private void updateSharedStorageFromServer() throws CacheException
	{
		// TODO: Implment SS cache update support.
	}
	
	/**
	 * Swaps out the current cache for an updated one, updates with the new information
	 *  puts the current cache into the old one, synchronized to avoid cache problems
	 *
	 * @param c New cache to swap in
	 */
	protected void swap(Cache c)
	{
		synchronized(this)
		{
			synchronized(c)
			{
				int tState = this.state;
				this.state = c.state;
				c.state = tState;
				
				Map tMap = this.groupList;
				this.groupList = c.groupList;
				c.groupList = tMap;
				
				tMap = this.guestList;
				this.guestList = c.guestList;
				c.guestList = tMap;
				
				LinkedList tList = this.networkList;
				this.networkList = c.networkList;
				c.networkList = tList;
				
				tList = this.sharedStorageList;
				this.sharedStorageList = c.sharedStorageList;
				c.sharedStorageList = tList;
				
				ZVMTasks tTasks = this.tasks;
				this.tasks = c.tasks;
				c.tasks = tTasks;
			}
		}
	}
}
