/*
 * dynamo.c
 * Author: Jennifer Hunt (jenhunt@us.ibm.com)
 * 
 * This program provides the functions for the Dynamic Memory Object (DynaMO) Manager
 *
 * DynaMO actions:
 * 	dynamo create dcssname filename.dmo
 *		create a DCSS.  DCSSname specifies the name assigned to the dcss
 *              and filename.dmo is a user defined file that specifies the files 
 * 	        and directories to make up the DCSS.
 * 	dynamo create dcssname access
 *		create an unformatted DCSS.
 * 	dynamo update dcssname access filename.dmo
 * 		update a dcss by replacing the contents with files and directories
 * 		in filename.dmo
 * 	dynamo remove dcss
 * 		permanently remove a DCSS
 * 	dynamo query 
 * 		query existing dcss's that are managed by dynamo.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include "dynamo.h"

/* Global Variables */
char *password;
char userid[9];
char SMAPI_IP[16];
/* 
 * Main function
 *
 * 	Parses command line inputs and takes appropriate action
 */
main (int argc, char *argv[]){
	char command[9];
	char dcssname[9];
	char dcssnamein[16];
	char access_file[50];
        char restrict_file[50];
	char dcss_size[9];
	char *result;

	int cmd,i;
	int rc = RC_OK;
	int unformatted = 0;

	#ifdef DEBUG
		fprintf(stderr,"---> Enter main\n");
	#endif 
	
	/* Initialize filenames */
	memset(access_file, '\0', sizeof(access_file));
	memset(dcssname, '\0', sizeof(dcssname));
        memset(restrict_file, '\0', sizeof(restrict_file));

	/* parse command line parameters */
	/* Parse the Command */
	if (argc >=2){
		strncpy(command,argv[1],9);
	} 
	
	#ifdef DEBUG
		fprintf(stderr,"Command: %s\n",command);
	#endif
	
	/* Parse the DCSS Name */
	if (argc>=3){
		strncpy(dcssnamein,argv[2],15);
	
		#ifdef DEBUG
			fprintf(stderr,"dcssnamein: %s\n",dcssnamein);
		#endif

		if(strlen(dcssnamein)>8){
			rc = NAME_ERROR;
		}
		
		for(i=0;strlen(dcssnamein);i++){
			dcssname[i]=toupper(dcssnamein[i]);
		}

		#ifdef DEBUG
			fprintf(stderr,"dcssname: %s\n",dcssname);
		#endif
	}

	/* Parse the access type or file name */
	if (argc>=4){
		strncpy(access_file,argv[3],49);
		#ifdef DEBUG
                        fprintf(stderr,"access_filename: %s\n",access_file);
                #endif
	}


        /* Parse the restricted user file is it exists */
	if (argc>=5){
		strncpy(restrict_file,argv[4],49);
		#ifdef DEBUG
                        fprintf(stderr,"restrict_filename: %s\n",restrict_file);
                #endif
	}

	/* Set Command Value */
	if (strcmp(command,CREATE)==0)
		cmd=CRT;
	else if (strcmp(command,UPDATE)==0)
		cmd=UPD;
	else if (strcmp(command,REMOVE)==0)
		cmd=REM;
	else if (strcmp(command,QUERY)==0)
		cmd=QUE;
        else if (strcmp(command,RESTRICT)==0)
		cmd=RST;
        else if (strcmp(command,DELUSER)==0)
		cmd=DLU;
        else if (strcmp(command,ADDUSER)==0)
		cmd=ADU;
        else if (strcmp(command,QRYUSERS)==0)
		cmd=QUS;
	else {
		#ifdef DEBUG
			fprintf(stderr,"Bad Command\n");
		#endif
		rc = COMMAND_ERROR;
	}

	#ifdef DEBUG
        	fprintf(stderr,"cmd: %i\n",cmd);
        #endif
	
	/* Get the Server IP address            */
	if (rc==RC_OK)
		rc = getIPaddress();

	/* Authenticate User                       */
	/* Find User Name in /proc/sysinfo  userid */
	if (rc==RC_OK){
		obtain_userid(userid);

		#ifdef DEBUG
                	fprintf(stderr,"userid: %s\n",userid);
	        #endif

		/* Prompt for z/VM password 		*/
		password = getpass("Enter password:");	

		/* Check if user/password is valid 	*/
		rc = authenticate_user(userid,password);

		/* exit with error if not		*/
		if (rc !=0){
			#ifdef DEBUG
				fprintf(stderr,"Could not Authenticate User\n");
                		fprintf(stderr,"rc: %i\n",rc);
	        	#endif
		} else {
			/* Handle the Command Type 		*/
			switch (cmd){
				/* Create Unformatted or Formatted */
				case CRT:
					#ifdef DEBUG
						fprintf(stderr,"Create Option\n");
						fprintf(stderr,"filename: %s\n",access_file);
					#endif
					/* Fail if no name is specified */
					if (strlen(dcssname)==0){
						rc = NONAME_ERROR;
					} else {
						/* If there is a file create a formatted DCSS */
						for (i = 0; i < 6; i++){
							if (strcmp(TYPES[i],access_file)==0){
								unformatted = 1;
							}
						}
					
						if(!unformatted){
							rc = create_formatted(dcssname,access_file);
						/* Else create an unformatted DCSS */
						}else {
							/* prompt for size */
							printf("Enter Memory Size (in M): \n");
							scanf("%s", dcss_size);
							rc = create_unformatted(dcssname,dcss_size,access_file);
						}
					}
		
					/* If successful write dcssname to .dcss */
					if (rc==RC_OK){
						add_dcssname(dcssname);
					}
					break;
                                /* Create restricted Unformatted */
				case RST:
					#ifdef DEBUG
						fprintf(stderr,"restrict Option\n");
						fprintf(stderr,"filename: %s\n",access_file);
                                                fprintf(stderr,"filename: %s\n",restrict_file);
					#endif
					/* Fail if no name is specified */
					if (strlen(dcssname)==0){
						rc = NONAME_ERROR;
					} else {
						
						/* prompt for size */
						printf("Enter Memory Size (in M): \n");
						scanf("%s", dcss_size);
						rc = create_restricted(dcssname,dcss_size,access_file,restrict_file);
						
					}
		
					/* If successful write dcssname to .dcss */
					if (rc==RC_OK){
						add_dcssname(dcssname);
					}
					break; 
				/* Update a formatted DCSS */
				case UPD:
					if (strlen(dcssname)==0){
						rc = NONAME_ERROR;
					} else { 
						/* Purge the DCSS */
						rc = remove_dcss(dcssname);
					}

					/* Create the new formatted DCSS */
					if (rc==RC_OK){
						rc = create_formatted(dcssname,access_file);
						if (rc!=RC_OK)
							rc = UPDATE_ERROR;
					}

					/* If successful write dcssname to .dcss */
					if (rc==RC_OK){
						add_dcssname(dcssname);
					}
					break;
				/* Remove a DCSS */
				case REM:
					if (strlen(dcssname)==0){
	 	                               rc = NONAME_ERROR;
					} else {
						rc = remove_dcss(dcssname);
					}
					break;
                                /* de-authorize users from a restricted DCSS */
				case DLU:
					if (strlen(dcssname)==0){
	 	                               rc = NONAME_ERROR;
					} else {
						rc = add_users(dcssname,access_file,1);
					}
					break;
                                /* authorize users to a restricted DCSS */
				case ADU:
					if (strlen(dcssname)==0){
	 	                               rc = NONAME_ERROR;
					} else {
						rc = add_users(dcssname,access_file,0);
					}
					break;
				case QUE:
					rc = query();
			        	break;
                                case QUS:
                                        if (strlen(dcssname)==0){
	 	                               rc = NONAME_ERROR;
					} else {
					        rc = query_names(dcssname);
                                        }
			        	break;
				default:
					/* invalid option - will not get here, handled above. */
					break;
			}
		} /* end authenticated user */
	}
	/* Handle the RC message */
	char msgrc[9];
	sprintf(msgrc, "%d", rc);
	msg1(stdout, rc, msgrc);

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit main\n");
        #endif
}

/**
 *
 * add_dcssname
 *
 *	Add the dcss name to the list of managed DCSS's
 *
 */
int add_dcssname(char *dcssname){
	int rc = RC_OK;
	FILE *fp;

	#ifdef DEBUG
                fprintf(stderr,"---> Enter add_dcssname\n");
        #endif
	
	/* Add an endline to the end of the name */
	strcat(dcssname,"\n");

	/* Open the .dcss file for appending */
	if ((fp = fopen(DCSSFILE, "a"))==NULL){
		rc = FILE_ERROR;
	} else{
		fputs(dcssname,fp);
		fclose(fp);
	} 

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit add_dcssname\n");
        #endif

return rc;
}

/**
 * getIPaddress
 *
 * 	Get the server ip address from the config file.
 *
 */
int getIPaddress(){
	int rc = RC_OK;
	int found=0;
	FILE *fp;
	char line[81];
	char tag[9]="SMAPI_IP";

	#ifdef DEBUG
                fprintf(stderr,"---> Enter getIPaddress\n");
        #endif

	/* Open the config file, fail if it does not exist */
	if((fp = fopen(CONFIGFILE, "r"))==NULL){
		rc = CONFIG_ERROR;
	} else {
		/* Get a line from the config file */
		while(fgets(line,80,fp) != NULL){
			/* Compare the line to the tag */
			if(strncmp(tag,line,8)==0){
				strcpy(SMAPI_IP,line+9);
				found=1;
			}
		}

		/* Close the files */
		fclose(fp);

	}
	
	/* Check if we found the IP address */
	if (found==0){
		rc = CONFIG_ERROR;
		#ifdef DEBUG
			fprintf(stderr,"Could not find the IP address\n");
		#endif
	}

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit getIPaddress\n");
        #endif
return rc;
}

/**
 *
 * removed_dcssname
 *
 * 	Remove the dcss name from the list of managed DCSS's
 *
 */
int remove_dcssname(char *dcssname){
	int rc = RC_OK;
	int namelen;
	FILE *old_fp, *new_fp;
	char line[81];
	char tempfile[5]="temp";

	#ifdef DEBUG
                fprintf(stderr,"---> Enter remove_dcssname\n");
        #endif
	
	/* Determine the length of the dcssname */
	namelen = strlen(dcssname);
	dcssname[namelen]='\n';

	/* Open the .dcss file, fail if it does not exist */
	if(((old_fp = fopen(DCSSFILE, "r"))==NULL) || ((new_fp = fopen(tempfile,"w"))==NULL)){
		rc = FILE_ERROR;
	} else {
		/* Get a line from the .dcss file */
		while(fgets(line,80,old_fp) != NULL){
			/* Compare the line to the target name and write the name if no match */
			if(strncmp(dcssname,line,namelen+1)!=0){
				fputs(line,new_fp);	
			}
		}
		/* Close the files */
		fclose(old_fp);
		fclose(new_fp);

		/* Rename the new file */
		rc = rename(tempfile,DCSSFILE);
	}

	/* Replace the endline with a Null terminator */
	dcssname[namelen]='\0';

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit remove_dcssname\n");
        #endif
return rc;
}

/**
 *
 * adddel_username
 *
 * 	Add or delete a user from the list of users authorized to load this dcss
 *
 */
int adddel_username(char *dcssname, char *username, int del){
	int rc = RC_OK;
	int namelen;
	FILE *old_fp, *new_fp, *fp;
	char line[81];
	char tempfile[5]="temp";
        char myfilename[50] = ".dcss.";

        /* get full filename */
        strcat(myfilename, dcssname); 

	#ifdef DEBUG
                fprintf(stderr,"---> Enter adddel_username\n");
        #endif
        
        /* Determine the length of the username */
	namelen = strlen(username);
	username[namelen]='\n';
	
	

	/* Open the .dcss file, fail if it does not exist */
	if(((old_fp = fopen(myfilename, "r"))==NULL) || ((new_fp = fopen(tempfile,"w"))==NULL)){
		if (del > 0) rc = FILE_ERROR;
        
	} else {
              
                 
		/* Get a line from the .dcss file */
		while(fgets(line,80,old_fp) != NULL){
			/* Compare the line to the target name and write the name if no match */
			if(strncmp(username,line,namelen+1)!=0){
				fputs(line,new_fp);	
			}
		}
		/* Close the files */
		fclose(old_fp);
		fclose(new_fp);

		/* Rename the new file */
		rc = rename(tempfile,myfilename);

        }
                
        /* file now has specified name removed, check if we need to add it in */
        if (del == 0)
        {
          /* Open the appropriate file for appending */
          if ((fp = fopen(myfilename, "a"))==NULL){
	  rc = FILE_ERROR;
          } else{
          	  fputs(username,fp);
		  fclose(fp);
	  } 
        }

        /* Replace the endline with a Null terminator */
        username[namelen]='\0'; 
        

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit adddel_username\n");
        #endif
return rc;
}

/**
 *
 * add_users
 *
 *	Iterate through the list of users to add
 */
int add_users(char *dcssname, char *filename, int del){
	int rc = RC_OK;
	int lines = 0;
	int i;
	FILE *fp;
	char username[9];

	#ifdef DEBUG
                fprintf(stderr,"---> Enter add users\n");
        #endif
	
	/* Open the DCSS File */
	if ((fp = fopen(filename, "r"))==NULL){
	        rc = FILE_ERROR;
        } else{
		/* Display Query info  */
        	while (fgets(username,80,fp)){
			
			for(i=0; i<9; i++){
				if(username[i]=='\n')
					username[i]='\0';
			}
			
                        add_oneuser(dcssname, username, del); 
                        
		}
		
	}

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit add users\n");
        #endif

return rc;
}

/**
 * add_oneuser
 * 
 * 	authorize a user to acces a dcss 
 *
 * @param dcssname	Name of the DCSS 
 * @param username	Name of the user 
 *
 * @return 0 on success, non-zero on failure
 **/
int add_oneuser(char *dcssname, char *username, int del){
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char buffer[50];
        char return_buffer[50]={0};	
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter add_oneuser\n");
        #endif

	/* Set up the parameter list for the SMAPI call */
	if (del == 0) append_buffer(buffer+buffer_size, SSSMEMAN, &buffer_size);
        else append_buffer(buffer+buffer_size, SSSMEMDN, &buffer_size);

	/* Append userid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);	

	/* Append password */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append targetid */
	append_buffer(buffer+buffer_size, username, &buffer_size);

	/* Append DCSS name */
	append_buffer(buffer+buffer_size, dcssname, &buffer_size);
	
	/* Append the terminating null */
        buffer[buffer_size]=0x01;
        buffer[buffer_size+1]=0x01;
        buffer_size+=2;

	/* call add user SMAPI */
        rc = call_smapi(buffer, buffer_size,&rs,return_buffer);
	
        adddel_username(dcssname, username, del);
	
		
	#ifdef DEBUG
                fprintf(stderr,"<--- Exit add_oneuser\n");
        #endif

return rc;
}

/**
 *
 * query
 *
 *	Iterate through the list of existing DCSS's and print output
 */
int query(){
	int rc = RC_OK;
	int lines = 0;
	int i;
	FILE *fp;
	char dcssname[9];

	#ifdef DEBUG
                fprintf(stderr,"---> Enter query\n");
        #endif
	
	/* Open the DCSS File */
	if ((fp = fopen(DCSSFILE, "r"))==NULL){
	        rc = FILE_ERROR;
        } else{
		/* Display Query info  */
        	while (fgets(dcssname,80,fp)){
			/* Print header if first line */ 
			if (lines==0){
				output0(stdout, 1);
				lines++;
			}
			for(i=0; i<9; i++){
				if(dcssname[i]=='\n')
					dcssname[i]='\0';
			}
			query_dcss(dcssname);
		}
		/* Display a message if no dcss's exist */
		if(lines==0)
			rc = NODCSS_ERROR;	
	}

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit query\n");
        #endif

return rc;
}

/**
 *
 * query_names
 *
 *	print names of all users currently authorized for specified dcss
 */
int query_names(char *dcssname){
	int rc = RC_OK;
	int lines = 0;
	int i;
	FILE *fp;
        char myfilename[50] = ".dcss.";
        char username[9];
	
        
	#ifdef DEBUG
                fprintf(stderr,"---> Enter query_names\n");
        #endif

        strcat(myfilename, dcssname);
	
	/* Open the appropriate File */
	if ((fp = fopen(myfilename, "r"))==NULL){
	        rc = FILE_ERROR;
        } else{
		/* Display Query info  */
        	while (fgets(username,80,fp)){
			
			for(i=0; i<9; i++){
				if(username[i]=='\n')
					username[i]='\0';
			}
			printf("%s \n", username);
		}
		
	}

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit query_names\n");
        #endif

return rc;
}

/**
 * query_dcss
 * 
 * 	query an existing DCSS 
 *
 * @param dcssname	Name of the DCSS to be queried
 *
 * @return 0 on success, non-zero on failure
 **/
int query_dcss(char *dcssname){
	int rc = RC_OK;
	int rs = 0;
	int i,delm;
	int buffer_size = 0;
	char buffer[50];
	char return_buffer[256]={0};
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter query_dcss\n");
        #endif

	/* Set up the parameter list for the SMAPI call */
	append_buffer(buffer+buffer_size, DMSCSSSQ, &buffer_size);

	/* Append userid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);	

	/* Append password */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append targetid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);

	/* Append DCSS name */
	append_buffer(buffer+buffer_size, dcssname, &buffer_size);
	
	/* Append the terminating null */
        buffer[buffer_size]=0x01;
        buffer[buffer_size+1]=0x01;
        buffer_size+=2;

	/* call Shared Storage Query SMAPI */
        rc = call_smapi(buffer, buffer_size,&rs,return_buffer);	
	
	/* Display the query result */
	delm=0;
	if (rc==0){
                for(i=0; i < 256; i++){
			if(delm==2)
				fputc(return_buffer[i], stderr);	
			if(return_buffer[i]==0x00){
				delm++;
				i+=3;
			}
		}
					
                fprintf(stderr,"\n");
	}else{
		rc = QUERY_ERROR;
		#ifdef DEBUG
			fprintf(stdout,"Query failed\n");
		#endif
	}
	
	#ifdef DEBUG
                fprintf(stderr,"<--- Exit query_dcss\n");
        #endif

return rc;
}
	
/**
 * remove_dcss
 * 
 * 	Purge an existing DCSS
 *
 * @param dcssname	Name of the DCSS to be purged
 *
 * @return 0 on success, non-zero on failure
 **/
int remove_dcss(char *dcssname){
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char buffer[50];
	char return_buffer[50]={0};
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter remove_dcss\n");
        #endif

	/* Set up the parameter list for the SMAPI call */
	append_buffer(buffer+buffer_size, SSSMEMDL, &buffer_size);

	/* Append userid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);	

	/* Append password */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append targetid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);

	/* Append DCSS name */
	append_buffer(buffer+buffer_size, dcssname, &buffer_size);
	
	/* Append the terminating null */
        buffer[buffer_size]=0x01;
        buffer[buffer_size+1]=0x01;
        buffer_size+=2;

	/* call  Shared Storage Delete SMAPI */
        rc = call_smapi(buffer, buffer_size,&rs,return_buffer);	

	/* If successful, remove from the list of DCSS's 			 */
	/* rc 600, rs 16 indicates a pending purge, allow the dcss to be removed */
	if((rc==RC_OK)|| (rc==600 && rs==16)){
		rc = remove_dcssname(dcssname);		
	} else {
		if (rc==600)
			rc = NOTFOUND_ERROR;
		else
			rc = REMOVE_ERROR;
	}

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit remove_dcss\n");
        #endif

return rc;
}
	
/**
 * create_segment
 *
 *	Define and save a Segment
 *
 * @param dcssname	Name of the DCSS to be created
 * @param size          Size of the DCSS to be created
 * @param access_type   Access type of the DCSS to be created
 * @param rstd          whether or not this DCSS should be restricted
 *
 * @return 0 on sucess, non-zero on failure
 **/
int create_segment(char *dcssname,char *page_range, char *access_type, int rstd)
{
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char buffer[50];
	char return_buffer[50]={0};
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter create_segment\n");
        #endif
	
	/* Set up the parameter list for the SMAPI call */
	if (rstd == 0) append_buffer(buffer+buffer_size, SSSMEMC2, &buffer_size);
        else append_buffer(buffer+buffer_size, SSSMEMRS, &buffer_size);

	/* Append userid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);		

	/* Append Password to the parameter list */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append targetid (userid) */
	append_buffer(buffer+buffer_size, userid, &buffer_size);		
		
	/* Append dcss name */
	append_buffer(buffer+buffer_size, dcssname, &buffer_size);

	/* Append page range */
	append_buffer(buffer+buffer_size, page_range, &buffer_size);
	
	/* Append Access Type */
	append_buffer(buffer+buffer_size, access_type, &buffer_size);
	
	/* Append the terminating null */
        buffer[buffer_size]=0x01;
        buffer[buffer_size+1]=0x01;
        buffer_size+=2;

	/* Call Shared Storage Create SMAPI */
	rc = call_smapi(buffer, buffer_size, &rs, return_buffer);

	#ifdef DEBUG
		fprintf(stderr,"Back from Call_Smapi\n");
		fprintf(stderr,"   return_code: %d\n",rc);
		fprintf(stderr,"   reason_code: %d\n",rs);
		fprintf(stderr,"   return_buffer: %s\n",return_buffer);
        #endif
	
	#ifdef DEBUG
	        fprintf(stderr,"<--- Exit create_segment\n");
	#endif
return rc;
}
		
/**
 * create_unformatted
 *
 *	Creates an unformatted DCSS
 *
 * @param dcssname	Name of the DCSS to be created
 * @param size          Size of the DCSS to be created
 * @param access_type   Access type of the DCSS to be created
 *
 * @return 0 on sucess, non-zero on failure
 **/
int create_unformatted(char *dcssname,char *size, char *access_type)
{
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char buffer[50];
	char return_buffer[50]={0};
	char page_range[18];
	char access[3];
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter create_unformatted\n");
		fprintf(stderr,"size: %s\n", size);
		fprintf(stderr,"use_gaps: %s\n",use_gaps);
        #endif

	/* Choose the location of the DCSS */
	rc = choose_location(size, page_range, use_gaps);
	
	if (rc == 0){	
		/* Create the DCSS */
                
		rc = create_segment(dcssname, page_range, access_type, 0);
		
	}  else {
		#ifdef DEBUG
			fprintf(stderr,"Error with Choose Location\n");
		#endif
	}
	
	#ifdef DEBUG
	        fprintf(stderr,"<--- Exit create_unformatted\n");
	#endif
return rc;
}		

/**
 * create_restricted
 *
 *	Creates an restricted DCSS
 *
 * @param dcssname	Name of the DCSS to be created
 * @param size          Size of the DCSS to be created
 * @param access_type   Access type of the DCSS to be created
 * @param filename      file containing users to be allowed to access DCSS
 *
 * @return 0 on sucess, non-zero on failure
 **/
int create_restricted(char *dcssname,char *size, char *access_type, char *filename)
{
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char buffer[50];
	char return_buffer[50]={0};
	char page_range[18];
	char access[3];
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter create_restricted\n");
		fprintf(stderr,"size: %s\n", size);
		fprintf(stderr,"filename: %s\n",filename);
        #endif

	/* Choose the location of the DCSS */
	rc = choose_location(size, page_range, use_gaps);
	
	if (rc == 0){	
		/* Create the DCSS */
                
		rc = create_segment(dcssname, page_range, access_type, 1);
                add_users(dcssname, filename, 0);
		
	}  else {
		#ifdef DEBUG
			fprintf(stderr,"Error with Choose Location\n");
		#endif
	}
	
	#ifdef DEBUG
	        fprintf(stderr,"<--- Exit create_restricted\n");
	#endif
return rc;
}		

/**
 * choose_location
 *
 * 	Issues the SMAPI call to choose a location of a DCSS based on the size and
 * 	model user.

 * @param size 	Size of the DCSS to be created
 * @param page_range The page range to use for the DCSS
 **/
int choose_location(char *dcss_size, char *page_range, char *use_gaps)
{
	int rc = RC_OK;
	int rs = 0;
	int buffer_size = 0;
	char *floor = "0";
	char buffer[50];
        char return_buffer[50]={0};

	#ifdef DEBUG
                fprintf(stderr,"---> Enter choose_location\n");
        #endif

	/* Set up parameter list for SMAPI call */
	append_buffer(buffer+buffer_size, SSSFNDST, &buffer_size);

	/* Find User Name in /proc/sysinfo  userid */
	append_buffer(buffer+buffer_size, userid, &buffer_size);		

	/* Append Password to the parameter list */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append Target ID */
	append_buffer(buffer+buffer_size, userid, &buffer_size);

	/* Append SSSFNDST Parameters */
	/* Append the floor - 0 for us */
	append_buffer(buffer+buffer_size, floor, &buffer_size);

	/* Append the size */
	append_buffer(buffer+buffer_size, dcss_size, &buffer_size);

	/* Append if use gaps */
	append_buffer(buffer+buffer_size, use_gaps, &buffer_size);

	/* Append the model (our administrator) */
	append_buffer(buffer+buffer_size, userid, &buffer_size);

	/* Append the terminating null */
        buffer[buffer_size]=0x01;
        buffer[buffer_size+1]=0x01;
        buffer_size+=2;

	/* call SSSFNDST SMAPI */
	rc = call_smapi(buffer, buffer_size,&rs,return_buffer);
	
	#ifdef DEBUG
		fprintf(stderr,"Back from Call_Smapi\n");
		fprintf(stderr, "   return_code: %d\n", rc);
		fprintf(stderr, "   reason_code: %d\n", rs);
		fprintf(stderr, "   return_buffer: %s\n",return_buffer);
	#endif

	/* Set the page range */
	if (rc==0){
		strcpy(page_range, return_buffer);
		#ifdef DEBUG
			fprintf(stderr,"page_range: %s\n",page_range);
		#endif
	} else
		rc = STORAGE_ERROR;

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit choose_location\n");
        #endif

return rc;
}
 
/*
 * create_formatted
 *
 *	Creates a formatted DCSS by first creating an unformatted DCSS and then copying data
 *
 * @param dcssname	Name of the DCSS to be created
 * @param filenam	Name of the file that contains the names of the files to be included
 *
 * @return 0 on success, non-zero on failure
 **/
int create_formatted(char *dcssname, char *filename ) 
{
	int rc = RC_OK;
	char *dest_dir, *start_addr, *end_addr;
	char dcss_size[33];
	float size;
	char shell_cmd[20]="rm -rf ";
	char access_type[3]; 
	char page_range[18];

	#ifdef DEBUG
                fprintf(stderr,"---> Enter create_formatted\n");
        #endif

	/* Prepare the DCSS content and return size */
	rc = prepare_content(filename,dcss_size);

	if (rc!=0 || strcmp(dcss_size,"0")==0){
		#ifdef DEBUG
			fprintf(stderr,"PREPPARE CONTENT ERROR: rc=%d dcss_size=%s\n",rc,dcss_size);
		#endif
		rc = PREPARECONTENT_ERROR;
	} else {
		#ifdef DEBUG
			fprintf(stderr,"dcss_size: %s\n",dcss_size);
		#endif
	
		/* Choose the location of the DCSS */
		rc = choose_location(dcss_size, page_range, use_gaps);
		
		/* Create the DCSS */
		strcpy(access_type, SN);
		rc = create_segment(dcssname, page_range, access_type, 1);
                add_oneuser(dcssname, userid, 0);


		/* Copy Data to the DCSS using the DCSS Block Device Driver */
		if (rc==RC_OK){
			output0(stdout, 2);
			#ifdef DEBUG
				fprintf(stderr,"Created DCSS, copy the data.\n");
			#endif
			rc = copy_data(dcssname);
	
			if (rc==RC_OK){
				/* Do the final save */
				strcpy(access_type, FI);
				rc = create_segment(dcssname, page_range, access_type, 0);
			} else {
				rc = COPYDATA_ERROR;
				/* purge the DCSS here since we failed?*/
				rc = create_segment(dcssname, page_range, "PU", 0);
			}
			rc = umount_dcss(dcssname);
		} else{
			rc = CREATE_ERROR;
			#ifdef DEBUG
				fprintf(stderr,"Create failed\n");
			#endif
		}
	} /* end prepare content successful */	
	
	/* Remove the temp directory */
	strcat(shell_cmd, DMOTEMP);
	system(shell_cmd);

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit create_formatted\n");
        #endif

return rc;
}

/*
 * copy_data
 *
 *	Copy the data to the DCSS using the DCSS Block Device Driver
 *
 * @return rc	Return code
 **/
int copy_data(char *dcssname)
{
	int rc = RC_OK;
	char shell_cmd[20] = "./copydcss ";

	#ifdef DEBUG
                fprintf(stderr,"---> Enter copy_data\n");
        #endif
	
	/* Issue the bash script to copy the dcss data */
	strcat(shell_cmd, dcssname);
	rc = system(shell_cmd);	
	
	#ifdef DEBUG
                fprintf(stderr,"<--- Exit copy_data\n");
        #endif

return rc;
}

/*
 * umount_dcss
 *
 * 	unmount and purge the dcss
 *
 * @return rc 	Return code
 **/
int umount_dcss(char *dcssname)
{
	int rc = RC_OK;
	char shell_cmd[20] = "./umountdcss ";

	#ifdef DEBUG
		fprintf(stderr,"---> Enter umount_dcss\n");
	#endif

	/* Issue the bash script to umount and remove the dcss */
	strcat(shell_cmd, dcssname);
	rc = system(shell_cmd);

	#ifdef DEBUG
		fprintf(stderr,"<--- Exit umount_dcss\n");
	#endif

return rc;
}
/*
 * prepare_content
 * 
 * 	Preare the content for the DCSS by executing the copy script
 * 
 * @param filename	Name of the file that contains the DCSS content
 * @return size		Size of the files to be transfered (in K)
 **/
int prepare_content(char *filename, char *dcss_size)
{
	int rc = RC_OK;
	FILE * fp;
	char size_str[51];
	int size = 0;
	int tempsize;
	char shell_cmd[26] = "./preparedcss ";
	char shell_rm_cmd[20]="rm -rf ";

	#ifdef DEBUG
                fprintf(stderr,"---> Enter prepare_content\n");
        #endif

	/* Call the bash shell script to prepare the dcss content */
	strcat(shell_cmd, filename);
	rc = system(shell_cmd);

	#ifdef DEBUG
		fprintf(stderr,"RC from prepare dcss: %d\n",rc);
	#endif

	if(rc==RC_OK){
	        /* Open the sizefile */
	        if ((fp = fopen(SIZEFILE, "r"))==NULL){
			rc = SIZEFILE_ERROR;
			#ifdef DEBUG
				fprintf(stderr,"Could not open size file.\n");
			#endif
		} else {
	        	/* Get the size */
	        	fgets(size_str,50,fp);
			size= atoi(size_str);
	
	        	/* Close the file */
	        	fclose(fp);

			/* Size is returned in KB - adjust for metadata and convert to MB */
			#ifdef DEBUG
				fprintf(stderr,"size str from script: %s\n", size_str);
				fprintf(stderr,"size int from script: %d\n", size);
			#endif

			/* Add 10% for metadata */
			size *= 1.1;	
			
			/* Convert kb to MB     */
			tempsize = size / 1024;	
			
			/* Round up 		*/
			if (size % 1024 != 0 )
				size = tempsize + 1;   
			else
				size = tempsize;

			#ifdef DEBUG
				fprintf(stderr,"size in adjusted MB: %d\n",size);
			#endif

			/* Convert the size to a string */
			sprintf(dcss_size,"%d",size);
		} /* end valid size */
	
		/* Remove the size file */
		strcat(shell_rm_cmd, SIZEFILE);
	        system(shell_rm_cmd);
	}
	#ifdef DEBUG
                fprintf(stderr,"<--- Exit prepare_content\n");
        #endif

return rc;
}

/*
 * obtain_userid
 * 
 *	Determine the user id that we are executing from
 * @return userid	
 **/
void obtain_userid(char *name)
{
	FILE *sysinfo;
	char fileline[100];
	char *line;
	char keyword[10] = {"VM00 Name:"};

	#ifdef DEBUG
                fprintf(stderr,"---> Enter obtain_userid\n");
        #endif

	/* open the sysinfo file */
	sysinfo = fopen("/proc/sysinfo", "r");

	/* Find the VM Name Line and copy the name */
	do {
		line = fgets(fileline, 100, sysinfo);  

		if (line != NULL) {
			if (strncmp(line,keyword, 10) == 0){
				line+=22;
				strncpy(name,line,8);
			}
		}
	} while (line != NULL);
	
	/* Append a null terminator */
	name[9]= '\0';	

	/* Close the file */
	fclose(sysinfo);

	#ifdef DEBUG
                fprintf(stderr,"<--- Exit obtain_userid\n");
        #endif
}

/*
 * append_buffer
 * 	
 * 	Append NULL terminated strings to a parameter buffer
 *
 * @param buffer_ptr	Pointer to the current position in the buffer
 * @param src		String to be appended
 * @param buffer_size	Current size of the buffer
 **/
void append_buffer(char *buffer_ptr, char *src, int *buffer_size)
{
	int src_size;
	
	/* Determine the size of the source */
	src_size = strlen(src) + 1;
	
	/* Copy the src to the destination */
	memcpy(buffer_ptr, src, src_size);

	/* Increment the buffer pointer */
	buffer_ptr += src_size;

	/* Increment the buffer size */
	*buffer_size += src_size;

	#ifdef DEBUG
                fprintf(stderr,"In append_buffer.  Appended: %s\n",src);
        #endif
}
	
/*
 * authenticate_user
 *
 * 	Authenticate the user and password with the SMAPI server
 *
 * @param userid 	Userid to authenticate
 * @param password	Password for userid
 *
 * @return 0 on success, non-zero on failure
 **/
int authenticate_user(char *userid, char* password)
{
	int rc = RC_OK;
	int rs = 0;
	int buffer_size=0;
	int i;
	char buffer[50];
	char return_buffer[50]={0};
	
	#ifdef DEBUG
                fprintf(stderr,"---> Enter authenticate_user\n");
        #endif

	/* Append the NULL SMAPI Call */
	append_buffer(buffer+buffer_size, SMAPI_NULL, &buffer_size);

	/* Append the user id */
	append_buffer(buffer+buffer_size, userid, &buffer_size);
	
	/* Append the password */
	append_buffer(buffer+buffer_size, password, &buffer_size);

	/* Append target (userid) */
	append_buffer(buffer+buffer_size, userid, &buffer_size);
	
	/* Append the terminating null */
	buffer[buffer_size]=0x01;
	buffer[buffer_size+1]=0x01;
	buffer_size+=2;

	/* Call the SMAPI */
	rc = call_smapi(buffer, buffer_size,&rs, return_buffer);	

	#ifdef DEBUG
		fprintf(stderr,"Back from Call_Smapi\n");
		fprintf(stderr,"   Return_code: %d\n",rc);
		fprintf(stderr,"   Reason_code: %d\n",rs);
		fprintf(stderr,"   Return_buffer: %s\n",return_buffer);
                fprintf(stderr,"<--- Exit authenticate_user\n");
        #endif

return rc;
}

/*
 * call_smapi
 * 
 * 	Call the smapi server.
 *
 * @param buffer	The buffer of SMAPI parameters
 * @param return_buffer The return buffer from SMAPI
 *
 **/
int call_smapi(char *buffer, int buffer_size, int *rs, char *return_buffer)
{
	int i;
	int rc = RC_OK;
 	int fd,readmore=1;
	int bytes_read;
	int total_bytes=0;
	int rcrslen;
	char return_code[9];
	char reason_code[9];
	char read_buffer[MAX_BUFFER_SIZE+1];
	char str_buffer[MAX_BUFFER_SIZE+1]={0};
	struct sockaddr_in servaddr;

	memset(read_buffer, '\0', sizeof(read_buffer));

	#ifdef DEBUG
                fprintf(stderr,"---> Enter call_smapi\n");
		fprintf(stderr,"Buffer Size: %i\n",buffer_size);
		fprintf(stderr,"Buffer: \n");
        	for(i=0; i < buffer_size; i++)
                        fputc(buffer[i], stderr);
                fprintf(stderr,"\n");

	#endif

	/* Create Socket */
	fd = socket(AF_INET, SOCK_STREAM, 0);
	
	if (fd<0){
		#ifdef DEBUG
			fprintf(stderr, "socket() failed\n");
		#endif
		rc = SOCKET_ERROR;
	} else { /* socket worked */
		#ifdef DEBUG
			fprintf(stderr,"socket created\n");
		#endif

		/* Set up the sockaddr */
		memset(&servaddr, 0, sizeof(servaddr));
        	servaddr.sin_family = AF_INET;
        	servaddr.sin_port = htons(SMAPI_PORT);
        	servaddr.sin_addr.s_addr = inet_addr(SMAPI_IP);
	
		/* Connect to the SMAPI Server */
		rc = connect(fd,(struct sockaddr *)&servaddr, sizeof(servaddr));
		if (rc<0){
                	#ifdef DEBUG
                        	fprintf(stderr, "connect() failed\n");
                	#endif
                	rc = CONNECT_ERROR;
        	} else { /* Connect Worked */
			#ifdef DEBUG
	                        fprintf(stderr,"socket connected\n");
        	        #endif

			/* Send the SMAPI Call */
			rc = send(fd, buffer, buffer_size, 0);

			if (rc<0){
                		#ifdef DEBUG
                        		fprintf(stderr, "send() failed\n");
                		#endif
                		rc = SEND_ERROR;
        		} else { /* Send Worked */
				#ifdef DEBUG
		                        fprintf(stderr,"data sent on socket\n");
                		#endif

				/* Read the response */
				rc = 0;
				while (readmore == 1) {
					#ifdef DEBUG
                        			fprintf(stderr,"reading data\n");
			                #endif

					bytes_read = read(fd, read_buffer, MAX_BUFFER_SIZE);

					#ifdef DEBUG
						fprintf(stderr,"%i bytes read: readbuffer: ",bytes_read);
						for(i=0; i<bytes_read; i++)	
							fputc(read_buffer[i], stderr);
						fprintf(stderr,"\n");
					#endif

					/* If there was data and readdone!=0 */
					if ((bytes_read > 3) && (read_buffer[3]!=0)){
						/* Add the data to the return buffer */
						for(i=0; i<bytes_read; i++)
							str_buffer[total_bytes+i]= read_buffer[i];
						total_bytes+=bytes_read;
					} else
						/* There was no data or read done */
						readmore = 0;
				} /* End while readmore */
		
				#ifdef DEBUG
					fprintf(stderr,"total_bytes: %i\n",total_bytes);
					fprintf(stderr,"str Buffer: \n");
                			for(i=0; i < total_bytes; i++)
			                        fputc(str_buffer[i], stderr);
			                fprintf(stderr,"\n");

				#endif
	
				if (total_bytes > 0 ){
					/* Pick off the return and reason codes */
					strcpy(return_code,strtok(str_buffer," "));
					strcpy(reason_code,strtok(NULL," "));

					/* Find the length of the return and reason codes */
					rcrslen = (strlen(return_code))+(strlen(reason_code))+2;

					if (rcrslen<total_bytes){
						for(i=0; i<(total_bytes-rcrslen);i++)
							return_buffer[i]=str_buffer[rcrslen+i];
					} else
						return_buffer[0]='\0';

					/* Convert the RC and RS to integers */
					rc = atoi(return_code);
					*rs = atoi(reason_code);

					#ifdef DEBUG
						fprintf(stderr,"rc: %d\n",rc);
						fprintf(stderr,"rs: %d\n",*rs);
						fprintf(stderr,"return_buffer: \n");
						for(i=0; i<(total_bytes-rcrslen); i++)
							fputc(return_buffer[i],stderr);
					#endif

					/* Handle SMAPI Error */
					if(rc==120)
						rc = PASSWORD_ERROR;
					else if (rc == 100)
						rc = AUTHENTICATION_ERROR;
				} else {
					rc = SERVER_ERROR;
					#ifdef DEBUG
						fprintf(stderr, "No Data Read\n");
					#endif
				}
			} /* End Send Worked */	
		} /* End Connect Worked */
	} /* End Socket Worked */ 

	/* Close the socket */
	close(fd);
	
	#ifdef DEBUG
                fprintf(stderr,"<--- Exit call_smapi\n");
        #endif

return rc;
}

/* Error message handlers */
void msg0(FILE *stream, int n){
	fprintf(stream, MESSAGES[n]);
}

void msg1(FILE *stream, int n, char *p1){
	fprintf(stream, MESSAGES[n], p1);
}

void msg2(FILE *stream, int n, char *p1, char *p2){
	fprintf(stream, MESSAGES[n], p1, p2);
}

void msg3(FILE *stream, int n, char *p1, char *p2, char *p3){
	fprintf(stream, MESSAGES[n], p1, p2, p3);
}

/* Output handlers */
void output0(FILE *stream, int n){
	fprintf(stream, OUTPUT[n]);
}

void output1(FILE *stream, int n, char *p1){
	fprintf(stream, OUTPUT[n],p1);
}
